#include "options.inc"

      module cfc_module
c-----------------------------------------------------------------------
c     Module for CFC stuff
c-----------------------------------------------------------------------
      implicit none

      integer, parameter                :: cfc11dim=67
      real, dimension(cfc11dim)         :: f11n, f11s

!    Atmospheric, northern hemisphere CFC-11 concentrations,
!    67 years from 1931-1997 (by courtesy of Stephen Walker, 1998)

      data f11n /
     *      0.00,     0.00,     0.00,     0.00,     0.00
     *,     0.00,     0.00,     0.00,     0.01,     0.01
     *,     0.02,     0.02,     0.03,     0.04,     0.05
     *,     0.08,     0.14,     0.24,     0.42,     0.67
     *,     1.02,     1.53,     2.21,     3.07,     4.11
     *,     5.39,     6.84,     8.19,     9.47,    11.13
     *,    13.38,    16.23,    19.72,    23.87,    28.57
     *,    33.78,    39.63,    46.27,    53.96,    62.75
     *,    72.36,    83.04,    95.27,   108.81,   121.86
     *,   134.47,   146.47,   157.18,   166.67,   175.11
     *,   182.81,   189.88,   197.72,   206.34,   216.12
     *,   227.48,   239.67,   251.05,   257.28,   263.65
     *,   266.33,   266.00,   267.58,   267.24,   266.38
     *,   265.25,   263.84 /

!    Atmospheric, southern hemisphere CFC-11 concentrations,
!    67 years from 1931-1997 (by courtesy of Stephen Walker, 1998)

      data f11s /
     *      0.00,     0.00,     0.00,     0.00,     0.00
     *,     0.00,     0.00,     0.00,     0.00,     0.01
     *,     0.01,     0.02,     0.02,     0.03,     0.04
     *,     0.05,     0.08,     0.14,     0.25,     0.42
     *,     0.66,     1.01,     1.49,     2.13,     2.93
     *,     3.92,     5.11,     6.39,     7.63,     8.96
     *,    10.63,    12.78,    15.47,    18.75,    22.61
     *,    26.99,    31.91,    37.46,    43.80,    51.07
     *,    59.27,    68.34,    78.54,    90.04,   102.29
     *,   114.40,   126.28,   137.45,   147.60,   158.08
     *,   166.54,   174.69,   183.03,   190.95,   200.50
     *,   209.82,   220.60,   231.46,   240.38,   249.23
     *,   254.53,   258.80,   260.56,   261.73,   261.29
     *,   261.08,   260.16 /

       integer :: cfc11_index           ! interpolation index


      integer, parameter                :: cfc12dim=67
      real, dimension(cfc12dim)         :: f12n, f12s

! Atmospheric northern hemisphere CFC-12 concentrations,
! 67 years from 1931-1997 (by courtesy of Stephen Walker, 1998)

      data f12n /
     *      0.00,     0.01,     0.02,     0.03,     0.04
     *,     0.07,     0.11,     0.17,     0.25,     0.37
     *,     0.53,     0.72,     0.94,     1.25,     1.65
     *,     2.33,     3.42,     4.73,     6.10,     7.58
     *,     9.23,    10.94,    12.82,    14.97,    17.40
     *,    20.22,    23.45,    26.86,    30.58,    35.03
     *,    40.12,    45.89,    52.66,    60.57,    69.53
     *,    79.46,    90.60,   103.16,   117.17,   132.52
     *,   149.00,   166.85,   186.59,   208.14,   229.37
     *,   249.51,   268.67,   286.35,   303.74,   320.96
     *,   337.33,   352.01,   367.69,   383.86,   401.25
     *,   420.96,   441.49,   463.22,   481.80,   493.96
     *,   504.64,   512.87,   520.12,   526.58,   531.59
     *,   534.81,   537.67  /

! Atmospheric southern hemisphere CFC-12 concentrations,
! 67 years from 1931-1997 (by courtesy of Stephen Walker, 1998)

      data f12s /
     *      0.00,     0.01,     0.01,     0.02,     0.03
     *,     0.04,     0.07,     0.11,     0.17,     0.25
     *,     0.36,     0.51,     0.69,     0.91,     1.21
     *,     1.64,     2.33,     3.32,     4.48,     5.77
     *,     7.19,     8.72,    10.37,    12.20,    14.25
     *,    16.59,    19.28,    22.28,    25.52,    29.18
     *,    33.42,    38.25,    43.80,    50.28,    57.76
     *,    66.21,    75.67,    86.31,    98.25,   111.52
     *,   126.02,   141.70,   158.82,   177.62,   197.54
     *,   217.31,   236.49,   254.72,   272.00,   290.34
     *,   306.90,   323.74,   341.15,   357.80,   376.45
     *,   394.00,   412.82,   433.46,   452.97,   470.55
     *,   484.07,   495.65,   504.49,   512.31,   518.39
     *,   523.50,   528.18  /

      integer :: cfc12_index           ! interpolation index

      end module cfc_module



      subroutine cfc11_init()
c-----------------------------------------------------------------------
c     Setup CFC11 as a passive tracer
c-----------------------------------------------------------------------
      use spflame_module
      use passive_tracer_module
      use cfc_module
      implicit none
      type( time_type) :: start
      type( time_type), allocatable :: interval(:)
      integer :: k,ref_year,io

      if (my_pe==0) then
        print*,' '
        print*,' initializing CFC11 for as a tracer '
        print*,' '
      endif

      if (my_pe==0) then
       print*,' read: reference year for CFC11 ',ref_year_cfc11
       print*,' read: CFC11 shall be tracer # ',n_cfc11 
       if (n_cfc11  > nt) then
        print*,' '
        print*,' WARNING WARNING WARNING WARNING'
        print*,'  number of tracer is only ',nt
        print*,'  CFC11 will not be integrated'
        print*,' WARNING WARNING WARNING WARNING'
        print*,' '
       endif
      endif

      if (n_cfc11  > nt)  call halt_stop(' in cfc11_init')


      start = set_date(ref_year_cfc11,1, 1, 0 , 0 ,0)

      allocate(interval(cfc11dim))

      if (get_calendar_type() == THIRTY_DAY_MONTHS ) then
       do k=1,cfc11dim
        call set_time(interval(k), 0, 360 )
       enddo
      elseif (get_calendar_type() == NO_LEAP ) then
       do k=1,cfc11dim
        call set_time(interval(k), 0, 365 )
       enddo
      else 
       print*,' wrong calendar type for CFC11'
       call halt_stop(' in passive_tracer_init')
      endif

      call init_forcing_interp(cfc11dim,.false.,
     &                         start,interval,cfc11_index)
      deallocate(interval)

      end subroutine cfc11_init



      subroutine cfc12_init()
c-----------------------------------------------------------------------
c     Setup CFC12 as a passive tracer
c-----------------------------------------------------------------------
      use spflame_module
      use passive_tracer_module
      use cfc_module
      implicit none
      type( time_type) :: start
      type( time_type), allocatable :: interval(:)
      integer :: k,ref_year,io

      if (my_pe==0) then
        print*,' '
        print*,' initializing CFC12 for as a tracer '
        print*,' '
      endif

      if (my_pe==0) then
       print*,' read: reference year for CFC12 ',ref_year_cfc12
       print*,' read: CFC12 shall be tracer # ',n_cfc12 
       if (n_cfc12  > nt) then
        print*,' '
        print*,' WARNING WARNING WARNING WARNING'
        print*,'  number of tracer is only ',nt
        print*,'  CFC12 will not be integrated'
        print*,' WARNING WARNING WARNING WARNING'
        print*,' '
       endif
      endif

      if (n_cfc11  > nt)  call halt_stop(' in cfc12_init')

      start = set_date(ref_year_cfc12,1, 1, 0 , 0 ,0)

      allocate(interval(cfc12dim))

      if (get_calendar_type() == THIRTY_DAY_MONTHS ) then
       do k=1,cfc12dim
        call set_time(interval(k), 0, 360 )
       enddo
      elseif (get_calendar_type() == NO_LEAP ) then
       do k=1,cfc12dim
        call set_time(interval(k), 0, 365 )
       enddo
      else 
       print*,' wrong calendar type for CFC12'
       call halt_stop(' in passive_tracer_init')
      endif

      call init_forcing_interp(cfc12dim,.false.,
     &                         start,interval,cfc12_index)
      deallocate(interval)

      end subroutine cfc12_init




    
      subroutine cfc_sflx(n,ident)
c-----------------------------------------------------------------------
c     apply surface fluxes for CFC11 or CFC12
c-----------------------------------------------------------------------
      use spflame_module
      use passive_tracer_module
      use cfc_module
      implicit none
      character (len=5) :: ident ! eihter CFC11 or CFC12
      integer         :: i,j,is,ie,js,je,n,p1,p2
      real, dimension(is_pe:ie_pe) :: kw,sc,eqconc
      real :: ff,fn,fs,weight,f1,f2

      is=max(is_pe,2); ie=min(ie_pe,imt-1)
      js=max(2,js_pe); je=min(je_pe,jmt-1)
      if (ident/='CFC11'.and.ident/='CFC12') 
     &            call halt_stop(' in cfc_sflx')
c
c     get interpolation factors 
c
      if (ident=='CFC11') call forcing_interp(cfc11_index,p1,p2,f1,f2)
      if (ident=='CFC12') call forcing_interp(cfc12_index,p1,p2,f1,f2)

      do j=js,je
c
c      interpolate between northern and southern hemisphere
c
       if (ident=='CFC11') then
            fs=f11s(p1)*f1+f11s(p2)*f2
            fn=f11n(p1)*f1+f11n(p2)*f2
       else
            fs=f12s(p1)*f1+f12s(p2)*f2
            fn=f12n(p1)*f1+f12n(p2)*f2
       endif

       if      ( yt(j) < -10.0 ) then; ff=fs
       else if ( yt(j) >  10.0 ) then; ff=fn
       else
            weight=(yt(j)+10.0)/20.0
            ff= weight*fn+(1.0-weight)*fs
       endif

       call wanninkhof(t(is,1,j,1,tau),stf_rest(is,j,n),
     &                 kw(is),ie-is+1)

       if (ident=='CFC11') then
        call schmidt_cfc11(t(is,1,j,1,tau),sc(is),ie-is+1)
        call eqconc_cfc_11(t(is,1,j,1,tau),t(is,1,j,2,tau), 
     &                     ff ,eqconc(is),ie-is+1)
       else
        call schmidt_cfc12(t(is,1,j,1,tau),sc(is),ie-is+1)
        call eqconc_cfc_12(t(is,1,j,1,tau),t(is,1,j,2,tau), 
     &                     ff ,eqconc(is),ie-is+1)
       endif


       do i=is,ie
        kw(i) = kw(i)*sqrt(660.0/sc(i))/3600.0
        stf(i,j,n)= kw(i) * (eqconc(i) -t(i,1,j,n,taum1) )
       enddo
      enddo

      end subroutine cfc_sflx


      subroutine eqconc_cfc_11(t,s,f11,eqconc,len)
c-----------------------------------------------------------------------
c     computes the equilibrium concentration of resolved cfc-11 
c     in pmol/l
c
c     t   = SST in degree celsius
c     s   = SSS in model units ( = [sal/1000] - 0.035 )
c     f11 = dry air mole fraction of cfc-11 in the atmosphere in PPTV
c           (parts per trillion of volume) (Dixon,Gammon,Thiele, 19??,
c           simulating chlorofluoromethane distribution in numerical
c           models of the world ocean)
c     alnf= solubility function in mol/(l*atm)
c     a,b = constants for the calculation of f11, taken from
c           Warner and Weiss, 1985: Solubilities of chloroflourocarbons
c            11 and 12 in water and seawater, Deep Sea Res, 32, No 12, 1485-1497
c
c     based on an idea by Ralf Doescher
c     implementation by Rene Redler, AWI, Dec 15, 1998
c-----------------------------------------------------------------------
      implicit none
      integer :: len,i
      real :: f11,tk, sal, alnf,a(4),b(3),t(len),s(len),eqconc(len)
      data a / -229.9261, 319.6552, 119.4471, -1.39165 /
      data b / -0.142382, 0.091459, -0.0157274 /
      do i=1,len
       tk=t(i)+273.2
       sal=(s(i)+0.035)*1000.0
       alnf= a(1)
     &    +(a(2)*(100./tk))+(a(3)*alog(tk/100.))+(a(4)*((tk/100.)**2))
     &    +sal*( b(1)+b(2)*(tk/100.)+b(3)*((tk/100.)**2) )
       eqconc(i)=f11*exp(alnf)
      enddo
      end subroutine eqconc_cfc_11


      subroutine eqconc_cfc_12(t,s,f12,eqconc,len)
c-----------------------------------------------------------------------
c     same but for CFC12
c-----------------------------------------------------------------------
      implicit none
      integer :: len,i
      real :: f12,tk, sal, alnf,a(4),b(3),t(len),s(len),eqconc(len)
      data a / -218.0971, 298.9702, 113.8049, -1.39165 /
      data b / -0.143566, 0.091015, -0.0153924 /
      do i=1,len
       tk=t(i)+273.2
       sal=(s(i)+0.035)*1000.0
       alnf= a(1)
     *    +(a(2)*(100./tk))+(a(3)*alog(tk/100.))+(a(4)*((tk/100.)**2))
     *    +sal*( b(1)+b(2)*(tk/100.)+b(3)*((tk/100.)**2) )

       eqconc(i)=f12*exp(alnf)
      enddo
      end subroutine eqconc_cfc_12


      subroutine schmidt_cfc11(tc,sc,len)
c-----------------------------------------------------------------------
c     calculates polynomial approximation to Schmidt number
c     for seawater (at salinity = 35 psu) after Wanninkhof (1992),
c     table A1
c-----------------------------------------------------------------------
      implicit none
      integer :: len,i
      real :: tc(len),sc(len) ! SST (input), schmidt number (output)
      do i=1,len
               sc(i) =    4039.8
     &                -   264.7     *tc(i)
     &                +     8.2552  *tc(i)*tc(i)
     &                -     0.10359 *tc(i)*tc(i)*tc(i)
      enddo
      end subroutine schmidt_cfc11

      subroutine schmidt_cfc12(tc,sc,len)
c-----------------------------------------------------------------------
c     same but for CFC11
c-----------------------------------------------------------------------
      implicit none
      integer :: len,i
      real :: tc(len),sc(len) ! SST (input), schmidt number (output)
      do i=1,len
               sc(i)=  3713.2  
     &                - 243.3      *tc(i)
     &                +   7.5879   *tc(i)*tc(i)
     &                -   0.095215 *tc(i)*tc(i)*tc(i)
      enddo
      end subroutine schmidt_cfc12


