#include "options.inc"

      module dens_module
c-----------------------------------------------------------------------
c
c     all stuff which deals with density, taken from MOM
c
c     five different kinds for the equation of state are
c     implemented so far:
c          linear e.o.s                   : linear_density
c          3.order fit from FLAME         : flame_density
c          McDougall Wright and Jacketts density : MWJ_density
c
c     originals authors:     k. dixon              e-mail=> kd@gfdl.gov
c                            FLAME authors
c     SPFLAME version:   c.eden
c
c                             jul 2001 c.eden
c-----------------------------------------------------------------------
      implicit none
c     public subroutines and functions in this module:
      public init_dens_module   ! subroutine, initialisation of e.o.s
                                ! must be called before any other call
                                ! of functions in this module
      public model_dens         ! subroutine, returns model density anomaly
      public model_dens_kvec    ! same as above but with k-ref as vector argument
      public model_dens_scalar  ! (scalar) function, returns model density anomaly
      public drhodt,drhods      ! subroutines, return partial derivatives
                                ! of model density with respect to
                                ! pot. temp. and salinity
      public drhodtt, drhodss
      public freezing_point     ! subroutine, returns freezing_point of
                                ! sea water as a function of salinity
                                ! stops if salinity is below zero
      real, parameter, public :: alpha_lin = -0.2e-3, beta_lin = 0.0e-3
      real, allocatable,private :: z_dens(:)
      integer,private           :: km_dens
#ifdef flame_density
      private flame_eq_of_state
      real :: dn_to(45),dn_so(45)
      real :: dn_coef(45,9)
#elif defined MWJ_density
c       original parameter: input degC,psu, dbar -> kg/m^3
c       real :: c0 =1.96637339e8, c1 =1.44604544e6
c       real :: c2 =-1.07691980e4 , c3 =8.52955498e1 
c       real :: c4 =5.79854265e5 , c5 =-1.47731528e3
c       real :: c6 =4.19489109e2 , c7 =2.00122349e3
c       real :: c8 =2.13851025e-2 , c9 =9.68774141e-1
c       real :: c10=-6.54785602e-3 , c11=-2.50468726e-6
c       real :: c12=6.28902345e-8 , c13=2.82414763e-9
c       real :: c14=1.96668928e5 , c15=1.43302205e3
c       real :: c16=-9.09231525 , c17=7.91654429e-2
c       real :: c18=3.98630534e-5 , c19=4.17831720e2
c       real :: c20=-1.87581316 , c21=-3.87902837e-5
c       real :: c22=1.00765828 , c23=3.12912597e-4
c       real :: rho0 = 1027.0
       ! check values -> ro = 1033.213242 kg/m^3 -rho0 = 6.213242 kg/m^3
       ! for       real :: t=20., s=35., p=2000.
c      changed: degC, psu dbar*100 -> g/cm^3
       real,parameter,private::c0 = 1.96637339e5,  c1 = 1.44604544e3
       real,parameter,private::c2 =-1.07691980e1 , c3 = 8.52955498e-2 
       real,parameter,private::c4 = 5.79854265e2 , c5 =-1.47731528e0
       real,parameter,private::c6 = 4.19489109e-1 ,c7 = 2.00122349e-2
       real,parameter,private::c8 = 2.13851025e-7 ,c9 = 9.68774141e-6
       real,parameter,private::c10=-6.54785602e-10,c11=-2.50468726e-13
       real,parameter,private::c12= 6.28902345e-15,c13= 2.82414763e-18
       real,parameter,private::c14= 1.96668928e5 , c15= 1.43302205e3
       real,parameter,private::c16=-9.09231525e0,  c17= 7.91654429e-2
       real,parameter,private::c18= 3.98630534e-5 ,c19= 4.17831720e2
       real,parameter,private::c20=-1.87581316e0,  c21=-3.87902837e-5
       real,parameter,private::c22= 1.00765828e0,  c23= 3.12912597e-4
       real,parameter,private::c24= 1e-2
       real,parameter, private :: rho0 = 1.027
#endif

      contains

      subroutine init_dens_module(zt,km,my_pe)
c-----------------------------------------------------------------------
c     initialize this module
c     input : zt(km)  depth of center of t-cells in cm
c-----------------------------------------------------------------------
      implicit none
      integer, intent(in) :: km,my_pe
      real,    intent(in) :: zt(km)    ! depth levels in cm 
      if (my_pe == 0) then
       print*,''
       print*,'Initialization of equation of state'
#if defined linear_density 
       print*,' using a linear equation of state'
       print*,' with drho/dT = ',alpha_lin
       print*,' and  drho/dS = ',beta_lin
#elif defined flame_density
       print*,' using the FLAME equation of state'
#elif defined MWJ_density
       print*,' using McDougall,Wright and Jackett equation of state'
       print*,' Note: this is the most accurate one'
#else
       print*,' you have to choose one of the density option'
       call halt_stop(' in init_dens_module')
#endif
      endif
      allocate( z_dens(km) )
      z_dens=zt   ! all in cm
      km_dens=km
#ifdef flame_density
       call flame_eq_of_state(dn_to,dn_so,dn_coef,km)
#endif
      if (my_pe == 0) print*,'done'
      end subroutine init_dens_module


      function model_dens_scalar(tqq,sqq,k
#ifdef partial_cell
     &                       ,ztp
#endif
     &                       )
c-----------------------------------------------------------------------
c     returns in situ density (pertubation) in g/cm^3 
c     input is pot. temperature (deg C) and salinity (psu-35)*1000
c     and pressure at level k
c-----------------------------------------------------------------------
      implicit none
      real                :: model_dens_scalar
      real ,intent(in)    :: tqq,sqq
      integer, intent(in) :: k
#ifdef partial_cell
      real, intent(in) :: ztp
      real  :: z(1)
#endif
      real :: t(1),s(1),ro(1)
      t=tqq;s=sqq
#ifdef partial_cell
      z=ztp
#endif
      call model_dens(t,s,ro,k,1
#ifdef partial_cell
     &                       ,z
#endif
     &                       )
      model_dens_scalar = ro(1)
      end function model_dens_scalar




      subroutine model_dens(tqq,sqq,ro,k,len
#ifdef partial_cell
     &                       ,ztp
#endif
     &                       )
c-----------------------------------------------------------------------
c     returns in situ density (pertubation) in g/cm^3 in ro
c     input is pot. temperature (deg C) and salinity (psu-35)*1000
c     and pressure at level k
c-----------------------------------------------------------------------
      implicit none
      integer, intent(in) :: k,len
      real ,intent(in),dimension(len)    :: tqq,sqq
#ifdef partial_cell
      real, intent(in) :: ztp(len)
#endif
      real,dimension(len),intent(out)    :: ro
      real                :: tq,sq
      integer             :: i

#ifdef linear_density

      do i=1,len
       ro(i) = beta_lin*sqq(i)+alpha_lin*tqq(i)
      enddo

#elif defined flame_density 

#ifdef partial_cell
      real :: c1,c2,c3,c4,c5,c6,c7,c8,c9,f1,f2
#endif
      do i=1,len
       tq=tqq(i)-dn_to(k)
       sq=sqq(i)-dn_so(k)
#ifdef partial_cell
c      it is assumed that for k=1 zt(k)==ztp(i,j)
       f2=(z_dens(k)-ztp(i))/
     &     max(2.,z_dens(k)-z_dens( max(1,k-1) ) )
       f1=1.-f2
       c1=f1*dn_coef(k,1) + f2*dn_coef( max(1,k-1) ,1)
       c2=f1*dn_coef(k,2) + f2*dn_coef( max(1,k-1) ,2)
       c3=f1*dn_coef(k,3) + f2*dn_coef( max(1,k-1) ,3)
       c4=f1*dn_coef(k,4) + f2*dn_coef( max(1,k-1) ,4)
       c5=f1*dn_coef(k,5) + f2*dn_coef( max(1,k-1) ,5)
       c6=f1*dn_coef(k,6) + f2*dn_coef( max(1,k-1) ,6)
       c7=f1*dn_coef(k,7) + f2*dn_coef( max(1,k-1) ,7)
       c8=f1*dn_coef(k,8) + f2*dn_coef( max(1,k-1) ,8)
       c9=f1*dn_coef(k,9) + f2*dn_coef( max(1,k-1) ,9)

       ro(i)  = (c1 +(c4 + c7*sq)*sq +
     &    (c3 + c8*sq + c6*tq)*tq)*tq +
     &    (c2 + (c5 + c9*sq)*sq)*sq
#else
       ro(i)  = (dn_coef(k,1) + 
     &    (dn_coef(k,4) + dn_coef(k,7)*sq)*sq +
     &    (dn_coef(k,3) + dn_coef(k,8)*sq 
     &   + dn_coef(k,6)*tq)*tq)*tq +
     &    (dn_coef(k,2) + (dn_coef(k,5) 
     &   + dn_coef(k,9)*sq)*sq)*sq
#endif
      enddo

#elif defined MWJ_density

       real :: poly1,poly2,t2,t3,s2,s32,p2,p3,t,s,p
       do i=1,len
        t=tqq(i);s=sqq(i)*1e3+35.
#ifdef partial_cell
        p = ztp(i)
#else
        p=z_dens(k)
#endif
        t2=t**2;t3=t**3;s2=s**2;s32=sqrt(s**3);p2=p**2;p3=p**3

        poly1=c0 +t*c1      +t2*c2     +t3*c3
     &           +s*c4      +s*t*c5    +s2*c6
     &           +p*c7      +p*t2*c8   +p*s*c9   +p2*c10
     &           +p2*t2*c11 +p2*t3*c12 +p3*t*c13

        poly2=c14 +t*c15 +t2*c16  +t3*c17   +t**4*c18
     &            +s*c19 +s*t*c20 +s*t3*c21 +s32*c22  +s32*t2*c23

        ro(i) = poly1/(poly2+c24*p) -rho0
       enddo
#endif
      end subroutine model_dens



      subroutine model_dens_kvec(tqq,sqq,ro,kvec,len
#ifdef partial_cell
     &                       ,ztp
#endif
     &                       )
c-----------------------------------------------------------------------
c     returns in situ density (pertubation) in g/cm^3 in ro
c     input is pot. temperature (deg C) and salinity (psu-35)*1000
c     and pressure at level k which is a vector argument
c-----------------------------------------------------------------------
      implicit none
      integer, intent(in) :: len
      real ,intent(in),dimension(len)    :: tqq,sqq
      integer, intent(in) :: kvec(len)
#ifdef partial_cell
      real, intent(in) :: ztp(len)
#endif
      real,dimension(len),intent(out)    :: ro
      real                :: tq,sq
      integer             :: i,k

#ifdef linear_density

      do i=1,len
       ro(i) = beta_lin*sqq(i)+alpha_lin*tqq(i)
      enddo

#elif defined flame_density 

#ifdef partial_cell
      real :: c1,c2,c3,c4,c5,c6,c7,c8,c9,f1,f2
#endif

      do i=1,len
       k=kvec(i)
       tq=tqq(i)-dn_to(k)
       sq=sqq(i)-dn_so(k)
#ifdef partial_cell
c      it is assumed that for k=1 zt(k)==ztp(i,j)
       f2=(z_dens(k)-ztp(i))/
     &     max(2.,z_dens(k)-z_dens( max(1,k-1) ) )
       f1=1.-f2
       c1=f1*dn_coef(k,1) + f2*dn_coef( max(1,k-1) ,1)
       c2=f1*dn_coef(k,2) + f2*dn_coef( max(1,k-1) ,2)
       c3=f1*dn_coef(k,3) + f2*dn_coef( max(1,k-1) ,3)
       c4=f1*dn_coef(k,4) + f2*dn_coef( max(1,k-1) ,4)
       c5=f1*dn_coef(k,5) + f2*dn_coef( max(1,k-1) ,5)
       c6=f1*dn_coef(k,6) + f2*dn_coef( max(1,k-1) ,6)
       c7=f1*dn_coef(k,7) + f2*dn_coef( max(1,k-1) ,7)
       c8=f1*dn_coef(k,8) + f2*dn_coef( max(1,k-1) ,8)
       c9=f1*dn_coef(k,9) + f2*dn_coef( max(1,k-1) ,9)

       ro(i)  = (c1 +(c4 + c7*sq)*sq +
     &    (c3 + c8*sq + c6*tq)*tq)*tq +
     &    (c2 + (c5 + c9*sq)*sq)*sq
#else
       ro(i)  = (dn_coef(k,1) + 
     &    (dn_coef(k,4) + dn_coef(k,7)*sq)*sq +
     &    (dn_coef(k,3) + dn_coef(k,8)*sq 
     &   + dn_coef(k,6)*tq)*tq)*tq +
     &    (dn_coef(k,2) + (dn_coef(k,5) 
     &   + dn_coef(k,9)*sq)*sq)*sq
#endif
      enddo

#elif defined MWJ_density

       real :: poly1,poly2,t2,t3,s2,s32,p2,p3,t,s,p
       do i=1,len
        k=kvec(i)
        t=tqq(i);s=sqq(i)*1e3+35.
#ifdef partial_cell
        p = ztp(i)
#else
        p=z_dens(k)
#endif
        t2=t**2;t3=t**3;s2=s**2;s32=sqrt(s**3);p2=p**2;p3=p**3

        poly1=c0 +t*c1      +t2*c2     +t3*c3
     &           +s*c4      +s*t*c5    +s2*c6
     &           +p*c7      +p*t2*c8   +p*s*c9   +p2*c10
     &           +p2*t2*c11 +p2*t3*c12 +p3*t*c13

        poly2=c14 +t*c15 +t2*c16  +t3*c17   +t**4*c18
     &            +s*c19 +s*t*c20 +s*t3*c21 +s32*c22  +s32*t2*c23

        ro(i) = poly1/(poly2+c24*p)  -rho0
       enddo
#endif
      end subroutine model_dens_kvec


      subroutine drhodt(tqq,sqq,dro,k,len
#ifdef partial_cell
     &                       ,ztp
#endif
     &                       )
!-----------------------------------------------------------------------
!     returns partial derivative of in situ density with respect
!     to pot. temperature in dro
!     input is pot. temperature (deg C) and salinity (psu-35)*1000
!     and pressure at level k
!-----------------------------------------------------------------------
      implicit none
      integer, intent(in) :: k,len
      real ,intent(in),dimension(len)    :: tqq,sqq
      real,dimension(len),intent(out)    :: dro
      integer             :: i
      real                :: tq,sq
#ifdef partial_cell
      real, intent(in) :: ztp(len)
#endif

#ifdef linear_density

      dro = alpha_lin

#elif defined flame_density 

#ifdef partial_cell
      real :: c1,c3,c4,c6,c7,c8,f1,f2
#endif

      do i=1,len
       tq=tqq(i)-dn_to(k); sq=sqq(i)-dn_so(k)
#ifdef partial_cell
c      it is assumed that for k=1 zt(k)==ztp(i,j)
       f2=(z_dens(k)-ztp(i))/
     &     max(2.,z_dens(k)-z_dens( max(1,k-1) ) )
       f1=1.-f2
       c1=f1*dn_coef(k,1) + f2*dn_coef( max(1,k-1) ,1)
       c3=f1*dn_coef(k,3) + f2*dn_coef( max(1,k-1) ,3)
       c4=f1*dn_coef(k,4) + f2*dn_coef( max(1,k-1) ,4)
       c6=f1*dn_coef(k,6) + f2*dn_coef( max(1,k-1) ,6)
       c7=f1*dn_coef(k,7) + f2*dn_coef( max(1,k-1) ,7)
       c8=f1*dn_coef(k,8) + f2*dn_coef( max(1,k-1) ,8)

       dro(i) = c1 + (c4 + c7*sq)*sq 
     &        + (2.0*c3 + 2.0*c8*sq 
     &        + 3.0*c6*tq)*tq
#else
       dro(i) = dn_coef(k,1) + (dn_coef(k,4) + dn_coef(k,7)*sq)*sq 
     &        + (2.0*dn_coef(k,3) + 2.0*dn_coef(k,8)*sq 
     &        + 3.0*dn_coef(k,6)*tq)*tq
#endif
      enddo

#elif defined MWJ_density

       real :: ro,poly1,poly2,t2,t3,s2,s32,p2,p3,t,s,p
       real :: dpoly1dt,dpoly2dt
       do i=1,len
        t=tqq(i);s=sqq(i)*1e3+35.
#ifdef partial_cell
        p = ztp(i)
#else
        p=z_dens(k)
#endif
        t2=t**2;t3=t**3;s2=s**2;s32=sqrt(s**3);p2=p**2;p3=p**3

        poly1=c0 +t*c1      +t2*c2     +t3*c3
     &           +s*c4      +s*t*c5    +s2*c6
     &           +p*c7      +p*t2*c8   +p*s*c9   +p2*c10
     &           +p2*t2*c11 +p2*t3*c12 +p3*t*c13

        poly2=c14 +t*c15 +t2*c16  +t3*c17   +t**4*c18
     &            +s*c19 +s*t*c20 +s*t3*c21 +s32*c22  +s32*t2*c23

        dpoly1dt= c1        +2*t*c2     +3*t2*c3
     &                      +s*c5   
     &                      +2*p*t*c8   
     &           +2*p2*t*c11 +3*p2*t2*c12 +p3*c13

        dpoly2dt= c15 +2*t*c16  +3*t2*c17   +4*t3*c18
     &             +s*c20 +3*s*t2*c21 +  2*s32*t*c23

        dro(i) = dpoly1dt/(poly2+c24*p)-poly1*dpoly2dt/(poly2+c24*p)**2
       enddo

#endif
      end subroutine drhodt



      subroutine drhodtt(tqq,sqq,dro,k,len
#ifdef partial_cell
     &                       ,ztp
#endif
     &                       )
!-----------------------------------------------------------------------
!     returns sec. partial derivative of in situ density with respect
!     to pot. temperature in dro
!     input is pot. temperature (deg C) and salinity (psu-35)*1000
!     and pressure at level k
!-----------------------------------------------------------------------
      implicit none
      integer, intent(in) :: k,len
      real ,intent(in),dimension(len)    :: tqq,sqq
      real,dimension(len),intent(out)    :: dro
      integer             :: i
      real                :: tq,sq
#ifdef partial_cell
      real, intent(in) :: ztp(len)
#endif

#ifdef linear_density
      dro = 0
#elif defined flame_density 
#ifdef partial_cell
       print*,' not yet implemented'
       call halt_stop(' in drhodtt')
#endif

      do i=1,len
       tq=tqq(i)-dn_to(k); sq=sqq(i)-dn_so(k)
c       dro(i) = dn_coef(k,1) + (dn_coef(k,4) + dn_coef(k,7)*sq)*sq 
c     &        + (2.0*dn_coef(k,3) + 2.0*dn_coef(k,8)*sq 
c     &        + 3.0*dn_coef(k,6)*tq)*tq
       dro(i) =   
     &   (2.0*dn_coef(k,3) + 2.0*dn_coef(k,8)*sq + 3.0*dn_coef(k,6)*tq)
     &  + 3.0*dn_coef(k,6)*tq
      enddo

#elif defined MWJ_density
       print*,' not yet implemented'
       call halt_stop(' in drhodtt')
#endif
      end subroutine drhodtt




      subroutine drhods(tqq,sqq,dro,k,len
#ifdef partial_cell
     &                       ,ztp
#endif
     &                       )
!-----------------------------------------------------------------------
!     returns partial derivative of in situ density with respect
!     to salinity in dro
!     input is pot. temperature (deg C) and salinity (psu-35)*1000
!     and pressure at level k
!-----------------------------------------------------------------------
      implicit none
      integer, intent(in) :: k,len
      real ,intent(in),dimension(len)    :: tqq,sqq
      real,dimension(len),intent(out)    :: dro
      integer             :: i
      real                :: tq,sq
#ifdef partial_cell
      real, intent(in) :: ztp(len)
#endif

#ifdef linear_density

      dro = beta_lin

#elif defined flame_density

#ifdef partial_cell
      real :: c2,c4,c5,c7,c8,c9,f1,f2
#endif

      do i=1,len
       tq=tqq(i)-dn_to(k); sq=sqq(i)-dn_so(k)
#ifdef partial_cell
c      it is assumed that for k=1 zt(k)==ztp(i,j)
       f2=(z_dens(k)-ztp(i))/
     &     max(2.,z_dens(k)-z_dens( max(1,k-1) ) )
       f1=1.-f2
       c2=f1*dn_coef(k,2) + f2*dn_coef( max(1,k-1) ,2)
       c4=f1*dn_coef(k,4) + f2*dn_coef( max(1,k-1) ,4)
       c5=f1*dn_coef(k,5) + f2*dn_coef( max(1,k-1) ,5)
       c7=f1*dn_coef(k,7) + f2*dn_coef( max(1,k-1) ,7)
       c8=f1*dn_coef(k,8) + f2*dn_coef( max(1,k-1) ,8)
       c9=f1*dn_coef(k,9) + f2*dn_coef( max(1,k-1) ,9)

       dro(i) = (c4 + 2.0*c7*sq 
     &   + c8*tq)*tq
     &   + c2 + (2.0*c5 + 3.0*c9*sq)*sq
#else
       dro(i) = (dn_coef(k,4) + 2.0*dn_coef(k,7)*sq 
     &   + dn_coef(k,8)*tq)*tq
     &   + dn_coef(k,2) + (2.0*dn_coef(k,5) + 3.0*dn_coef(k,9)*sq)*sq
#endif
      enddo

#elif defined MWJ_density
       real :: poly1,poly2,t2,t3,s2,s32,p2,p3,t,s,p
       real :: dpoly1ds,dpoly2ds,s12
       do i=1,len
        t=tqq(i);s=sqq(i)*1e3+35.
#ifdef partial_cell
        p = ztp(i)
#else
        p=z_dens(k)
#endif
        t2=t**2;t3=t**3;s2=s**2;s32=sqrt(s**3);p2=p**2;p3=p**3
        s12=sqrt(s)

        poly1=c0 +t*c1      +t2*c2     +t3*c3
     &           +s*c4      +s*t*c5    +s2*c6
     &           +p*c7      +p*t2*c8   +p*s*c9   +p2*c10
     &           +p2*t2*c11 +p2*t3*c12 +p3*t*c13

        poly2=c14 +t*c15 +t2*c16  +t3*c17   +t**4*c18
     &            +s*c19 +s*t*c20 +s*t3*c21 +s32*c22  +s32*t2*c23

        dpoly1ds=      
     &           +c4      +t*c5    +2*s*c6
     &                                 +p*c9   

        dpoly2ds=                                    
     &            +c19 +t*c20 +t3*c21 +1.5*s12*c22  +1.5*s12*t2*c23

        dro(i) =1e3*( dpoly1ds/(poly2+c24*p)-
     &                   poly1*dpoly2ds/(poly2+c24*p)**2)
       enddo
#endif
      end subroutine drhods


      subroutine drhodss(tqq,sqq,dro,k,len
#ifdef partial_cell
     &                       ,ztp
#endif
     &                       )
!-----------------------------------------------------------------------
!     returns sec. partial derivative of in situ density with respect
!     to salinity in dro
!     input is pot. temperature (deg C) and salinity (psu-35)*1000
!     and pressure at level k
!-----------------------------------------------------------------------
      implicit none
      integer, intent(in) :: k,len
      real ,intent(in),dimension(len)    :: tqq,sqq
      real,dimension(len),intent(out)    :: dro
      integer             :: i
      real                :: tq,sq
#ifdef partial_cell
      real, intent(in) :: ztp(len)
#endif

#ifdef linear_density
      dro = 0.0
#elif defined flame_density
#ifdef partial_cell
       print*,' not yet implemented'
       call halt_stop(' in drhodss')
#endif
      do i=1,len
       tq=tqq(i)-dn_to(k); sq=sqq(i)-dn_so(k)
c       dro(i) = (dn_coef(k,4) + 2.0*dn_coef(k,7)*sq 
c     &   + dn_coef(k,8)*tq)*tq
c     &   + dn_coef(k,2) + (2.0*dn_coef(k,5) + 3.0*dn_coef(k,9)*sq)*sq
       dro(i) = 2.0*dn_coef(k,7)*tq
     &   + (2.0*dn_coef(k,5) + 3.0*dn_coef(k,9)*sq)
     &   + 3.0*dn_coef(k,9)*sq
      enddo
#elif defined MWJ_density
       print*,' not yet implemented'
       call halt_stop(' in drhodss')
#endif
      end subroutine drhodss



      subroutine freezing_point(salt,frozen,imt)
c-----------------------------------------------------------------------
c       Returns the freezing point of seawater "tfreeze" [deg C]
c       as a function of model salinity "s" at pressure p=0.
c-----------------------------------------------------------------------
      implicit none
      integer, intent(in) :: imt
      real, intent(in)    :: salt(imt)
      real, intent(out)   :: frozen(imt)
      integer :: i
      real    :: s
      real, parameter :: a1=-0.0575, a2=1.710523e-3, a3=-2.154996e-4
      do i=1,imt
       s = 35.+1000.*salt(i)     ! convert from MOM-units to ppt
       frozen(i) = a1*s + a2*s*sqrt(s) + a3*s*s 
      enddo
      end subroutine freezing_point




#ifdef flame_density
      subroutine flame_eq_of_state(dn_to2,dn_so2,dn_coef2,km)
c-----------------------------------------------------------------------
c     returns coefficients and ref. temp/salinities
c     for the 3trd order polynomial fit to the eq. of state
c     Here for 45 levels used in the FLAME setup
c-----------------------------------------------------------------------
      implicit none
      integer n,km
      real :: dn_to2(km),dn_so2(km), dn_coef2(km,9)
c     normalized temperatures, salinities and coefficients
c     generated by program "eqstat" which fits 3rd order polynomials
c     to the equation of state for each model level.
      real :: dn_to(45),dn_so(45), dn_coef(45,9)
      data dn_to /
     &        14.3843011,14.2578683,14.0112627,13.6497450,13.5330309,
     &        13.2813462,13.0395064,12.7376460,12.4656092,12.1984710,
     &        11.7563071,11.2241432,10.6019889, 9.9197173, 9.1123623,
     &         8.4045991, 8.1903659, 8.0001810, 7.8492094, 7.6176854,
     &         7.2567335, 6.7512016, 6.0521144, 5.4176067, 5.2700249,
     &         5.1362931, 4.9912180, 4.7402026, 4.1307693, 3.0109479,
     &         1.9455630, 1.4232078, 1.3136686, 1.2719491, 1.2537283,
     &         1.2199199, 1.1951152, 1.2711003, 1.2248488, 1.1729706,
     &         1.1200841, 1.0762155, 0.2274399, 0.1929007, 0.1528538/
      data dn_so /
     &        -0.0075800,-0.0046650,-0.0011450,-0.0009550,-0.0006000,
     &        -0.0005550,-0.0003550,-0.0001700,-0.0000800,-0.0000100,
     &         0.0000500, 0.0001200,-0.0001400, 0.0000150, 0.0001200,
     &         0.0001900, 0.0001850, 0.0002700, 0.0003050, 0.0002700,
     &         0.0001900, 0.0001150, 0.0001100, 0.0001500, 0.0002000,
     &         0.0002350, 0.0002750, 0.0002800, 0.0002200, 0.0000450,
     &        -0.0000800,-0.0001200,-0.0001650,-0.0001800,-0.0001800,
     &        -0.0001750,-0.0002000,-0.0002000,-0.0002000,-0.0002000,
     &        -0.0001750,-0.0001650,-0.0002200,-0.0002100,-0.0002100/
      data (dn_coef(  1,n),n=1,9)/
     &         -.1999945E-03,0.7695597E+00,-.5027654E-05,-.2010208E-02,
     &         0.1662223E+00,0.3732400E-07,0.3964050E-02,0.3606369E-04,
     &         0.2171038E+01/
      data (dn_coef(  2,n),n=1,9)/
     &         -.2045001E-03,0.7706857E+00,-.4930207E-05,-.1992777E-02,
     &         0.2168222E+00,0.3599926E-07,0.3761402E-02,0.3549165E-04,
     &         0.1812958E+01/
      data (dn_coef(  3,n),n=1,9)/
     &         -.2090025E-03,0.7723355E+00,-.4824772E-05,-.1979757E-02,
     &         0.3649687E+00,0.3453949E-07,0.3597836E-02,0.3502083E-04,
     &         0.1523164E+01/
      data (dn_coef(  4,n),n=1,9)/
     &         -.2061498E-03,0.7730180E+00,-.4847599E-05,-.1998620E-02,
     &         0.3721007E+00,0.3499853E-07,0.3660466E-02,0.3546575E-04,
     &         0.1513641E+01/
      data (dn_coef(  5,n),n=1,9)/
     &         -.2059559E-03,0.7732896E+00,-.4841948E-05,-.2001704E-02,
     &         0.4135701E+00,0.3496045E-07,0.3661459E-02,0.3553309E-04,
     &         0.1490492E+01/
      data (dn_coef(  6,n),n=1,9)/
     &         -.2038953E-03,0.7737035E+00,-.4859710E-05,-.2015259E-02,
     &         0.4238350E+00,0.3532021E-07,0.3707557E-02,0.3586099E-04,
     &         0.1489936E+01/
      data (dn_coef(  7,n),n=1,9)/
     &         -.2022484E-03,0.7741502E+00,-.4870761E-05,-.2027064E-02,
     &         0.4477467E+00,0.3558170E-07,0.3742444E-02,0.3614380E-04,
     &         0.1479207E+01/
      data (dn_coef(  8,n),n=1,9)/
     &         -.2000107E-03,0.7747068E+00,-.4888063E-05,-.2042883E-02,
     &         0.4747043E+00,0.3596546E-07,0.3790085E-02,0.3652298E-04,
     &         0.1470456E+01/
      data (dn_coef(  9,n),n=1,9)/
     &         -.1978998E-03,0.7751624E+00,-.4905582E-05,-.2057476E-02,
     &         0.4828685E+00,0.3634790E-07,0.3835897E-02,0.3687892E-04,
     &         0.1467640E+01/
      data (dn_coef( 10,n),n=1,9)/
     &         -.1958142E-03,0.7755965E+00,-.4923089E-05,-.2071857E-02,
     &         0.4887710E+00,0.3673428E-07,0.3881095E-02,0.3723272E-04,
     &         0.1466076E+01/
      data (dn_coef( 11,n),n=1,9)/
     &         -.1920429E-03,0.7763693E+00,-.4958031E-05,-.2097773E-02,
     &         0.4850922E+00,0.3746791E-07,0.3961913E-02,0.3786143E-04,
     &         0.1467587E+01/
      data (dn_coef( 12,n),n=1,9)/
     &         -.1874184E-03,0.7773302E+00,-.5002062E-05,-.2130000E-02,
     &         0.4994565E+00,0.3839628E-07,0.4059349E-02,0.3864070E-04,
     &         0.1469846E+01/
      data (dn_coef( 13,n),n=1,9)/
     &         -.1812251E-03,0.7783590E+00,-.5069087E-05,-.2171760E-02,
     &         0.4333655E+00,0.3973514E-07,0.4193342E-02,0.3964842E-04,
     &         0.1495156E+01/
      data (dn_coef( 14,n),n=1,9)/
     &         -.1752861E-03,0.7796723E+00,-.5127606E-05,-.2215308E-02,
     &         0.4542648E+00,0.4100582E-07,0.4312595E-02,0.4069251E-04,
     &         0.1494881E+01/
      data (dn_coef( 15,n),n=1,9)/
     &         -.1679995E-03,0.7812382E+00,-.5203276E-05,-.2269527E-02,
     &         0.4251737E+00,0.4263636E-07,0.4456809E-02,0.4198542E-04,
     &         0.1500576E+01/
      data (dn_coef( 16,n),n=1,9)/
     &         -.1616836E-03,0.7825715E+00,-.5270308E-05,-.2317975E-02,
     &         0.4341092E+00,0.4411105E-07,0.4578771E-02,0.4313887E-04,
     &         0.1507324E+01/
      data (dn_coef( 17,n),n=1,9)/
     &         -.1604432E-03,0.7827295E+00,-.5280045E-05,-.2328938E-02,
     &         0.4228943E+00,0.4443989E-07,0.4602394E-02,0.4342388E-04,
     &         0.1510426E+01/
      data (dn_coef( 18,n),n=1,9)/
     &         -.1598681E-03,0.7827979E+00,-.5280071E-05,-.2336431E-02,
     &         0.4306736E+00,0.4462145E-07,0.4610658E-02,0.4363205E-04,
     &         0.1507186E+01/
      data (dn_coef( 19,n),n=1,9)/
     &         -.1597631E-03,0.7827048E+00,-.5274606E-05,-.2340319E-02,
     &         0.4284048E+00,0.4470532E-07,0.4610603E-02,0.4376191E-04,
     &         0.1506332E+01/
      data (dn_coef( 20,n),n=1,9)/
     &         -.1588770E-03,0.7827042E+00,-.5278651E-05,-.2350082E-02,
     &         0.3977782E+00,0.4499613E-07,0.4626064E-02,0.4403144E-04,
     &         0.1511353E+01/
      data (dn_coef( 21,n),n=1,9)/
     &         -.1565630E-03,0.7829823E+00,-.5297646E-05,-.2368438E-02,
     &         0.3341771E+00,0.4560839E-07,0.4668804E-02,0.4452262E-04,
     &         0.1520982E+01/
      data (dn_coef( 22,n),n=1,9)/
     &         -.1528735E-03,0.7835641E+00,-.5334075E-05,-.2398872E-02,
     &         0.2825812E+00,0.4659629E-07,0.4734220E-02,0.4528219E-04,
     &         0.1533388E+01/
      data (dn_coef( 23,n),n=1,9)/
     &         -.1472951E-03,0.7846405E+00,-.5396070E-05,-.2447318E-02,
     &         0.2109763E+00,0.4811526E-07,0.4827312E-02,0.4642522E-04,
     &         0.1545939E+01/
      data (dn_coef( 24,n),n=1,9)/
     &         -.1426251E-03,0.7855551E+00,-.5444288E-05,-.2488238E-02,
     &         0.1518860E+00,0.4941421E-07,0.4899482E-02,0.4741977E-04,
     &         0.1553680E+01/
      data (dn_coef( 25,n),n=1,9)/
     &         -.1433835E-03,0.7852385E+00,-.5426660E-05,-.2488701E-02,
     &         0.1364243E+00,0.4937288E-07,0.4875329E-02,0.4747849E-04,
     &         0.1551470E+01/
      data (dn_coef( 26,n),n=1,9)/
     &         -.1445448E-03,0.7847949E+00,-.5402167E-05,-.2485799E-02,
     &         0.1164959E+00,0.4923555E-07,0.4843527E-02,0.4747653E-04,
     &         0.1549515E+01/
      data (dn_coef( 27,n),n=1,9)/
     &         -.1459880E-03,0.7842619E+00,-.5372980E-05,-.2481922E-02,
     &         0.9095220E-01,0.4905440E-07,0.4804321E-02,0.4745857E-04,
     &         0.1547193E+01/
      data (dn_coef( 28,n),n=1,9)/
     &         -.1467579E-03,0.7838139E+00,-.5351539E-05,-.2484937E-02,
     &         0.6370349E-01,0.4909494E-07,0.4772692E-02,0.4760017E-04,
     &         0.1549039E+01/
      data (dn_coef( 29,n),n=1,9)/
     &         -.1442793E-03,0.7840179E+00,-.5372168E-05,-.2517144E-02,
     &         -.8473263E-02,0.5008149E-07,0.4784356E-02,0.4838619E-04,
     &         0.1562128E+01/
      data (dn_coef( 30,n),n=1,9)/
     &         -.1369815E-03,0.7852126E+00,-.5460129E-05,-.2594434E-02,
     &         -.1812696E+00,0.5252816E-07,0.4847993E-02,0.5012337E-04,
     &         0.1593518E+01/
      data (dn_coef( 31,n),n=1,9)/
     &         -.1313003E-03,0.7860641E+00,-.5527913E-05,-.2664084E-02,
     &         -.4257733E+00,0.5467702E-07,0.4855267E-02,0.5165746E-04,
     &         0.1619763E+01/
      data (dn_coef( 32,n),n=1,9)/
     &         -.1321362E-03,0.7854489E+00,-.5502435E-05,-.2678329E-02,
     &         -.6116769E+00,0.5501435E-07,0.4762733E-02,0.5201976E-04,
     &         0.1627586E+01/
      data (dn_coef( 33,n),n=1,9)/
     &         -.1376351E-03,0.7836986E+00,-.5408620E-05,-.2650093E-02,
     &         -.1064824E+01,0.5399455E-07,0.4615318E-02,0.5150764E-04,
     &         0.1626839E+01/
      data (dn_coef( 34,n),n=1,9)/
     &         -.1439258E-03,0.7817839E+00,-.5302371E-05,-.2614551E-02,
     &         -.1760482E+01,0.5271980E-07,0.4459424E-02,0.5085086E-04,
     &         0.1622689E+01/
      data (dn_coef( 35,n),n=1,9)/
     &         -.1504625E-03,0.7798217E+00,-.5192001E-05,-.2576500E-02,
     &         -.2033142E+01,0.5134681E-07,0.4301951E-02,0.5014581E-04,
     &         0.1617248E+01/
      data (dn_coef( 36,n),n=1,9)/
     &         -.1568052E-03,0.7779108E+00,-.5084237E-05,-.2540015E-02,
     &         -.2213485E+01,0.5000661E-07,0.4146716E-02,0.4947221E-04,
     &         0.1612021E+01/
      data (dn_coef( 37,n),n=1,9)/
     &         -.1631175E-03,0.7759765E+00,-.4977007E-05,-.2502900E-02,
     &         -.2901257E+01,0.4864564E-07,0.3994086E-02,0.4878341E-04,
     &         0.1609185E+01/
      data (dn_coef( 38,n),n=1,9)/
     &         -.1704176E-03,0.7738130E+00,-.4854077E-05,-.2455494E-02,
     &         -.2631662E+01,0.4695258E-07,0.3840168E-02,0.4790452E-04,
     &         0.1603177E+01/
      data (dn_coef( 39,n),n=1,9)/
     &         -.1764672E-03,0.7719631E+00,-.4750284E-05,-.2420605E-02,
     &         -.3039885E+01,0.4562383E-07,0.3690921E-02,0.4726087E-04,
     &         0.1599481E+01/
      data (dn_coef( 40,n),n=1,9)/
     &         -.1824207E-03,0.7701358E+00,-.4647738E-05,-.2386295E-02,
     &         -.3500498E+01,0.4430055E-07,0.3542942E-02,0.4662770E-04,
     &         0.1596121E+01/
      data (dn_coef( 41,n),n=1,9)/
     &         -.1883801E-03,0.7683288E+00,-.4544435E-05,-.2351805E-02,
     &         -.2655467E+01,0.4295357E-07,0.3395892E-02,0.4599620E-04,
     &         0.1591153E+01/
      data (dn_coef( 42,n),n=1,9)/
     &         -.1943374E-03,0.7665047E+00,-.4441356E-05,-.2316748E-02,
     &         -.2674579E+01,0.4158187E-07,0.3250562E-02,0.4535011E-04,
     &         0.1587437E+01/
      data (dn_coef( 43,n),n=1,9)/
     &         -.1930961E-03,0.7665301E+00,-.4440636E-05,-.2354018E-02,
     &         -.3304883E+01,0.4245507E-07,0.3093609E-02,0.4607462E-04,
     &         0.1598992E+01/
      data (dn_coef( 44,n),n=1,9)/
     &         -.1992031E-03,0.7646758E+00,-.4334472E-05,-.2317254E-02,
     &         -.3419570E+01,0.4100160E-07,0.2943317E-02,0.4538809E-04,
     &         0.1595076E+01/
      data (dn_coef( 45,n),n=1,9)/
     &         -.2051813E-03,0.7628414E+00,-.4230296E-05,-.2281238E-02,
     &         -.3650115E+01,0.3956332E-07,0.2794885E-02,0.4471275E-04,
     &         0.1592273E+01/
      if (km /=45 ) then
        print*,' Number of levels do not match FLAME grid'
        print*,' km=',km,' but must be 45'
        call halt_stop(' in init_eq_of_state')
      endif
      dn_to2=dn_to
      dn_so2=dn_so
      dn_coef2=dn_coef
      end subroutine flame_eq_of_state
#endif
      end module dens_module



#ifdef drive_test_dens

      subroutine diag_eq_of_state(file,km,zt)
c    writes a NetCDF file  with TS diagrams of density,etc
      use timing_module
      use dens_module
      implicit none
      integer :: km
      real :: zt(km)
      character (len=*) :: file
#ifdef netcdf_diagnostics
      integer :: imt,jmt,i,j,k
      real :: trange(3) = (/-5.,30.,0.5/)
      real :: srange(3) = (/15.,38.,0.5/) ! start, end and spacing in TS space
      real, allocatable :: t(:),s(:),rho(:,:),ztp(:),ss(:)
      integer :: ncid,iret,tdim,sdim,tid,sid,rid,zdim,zid
      integer :: rrid,rtid,rsid,drdtid,drdsid
      integer :: dims(3),start(3),count(3)
      character(len=80) :: name,unit
#ifdef netcdf_real4
      real (kind=4) :: x4(km)
      real (kind=4), allocatable :: buf(:,:)
#else
      real          :: x4(km)
      real         , allocatable :: buf(:,:)
#endif
#include "netcdf.inc"
      imt=(trange(2)-trange(1))/trange(3)
      jmt=(srange(2)-srange(1))/srange(3)
      allocate( t(imt), s(jmt), ss(imt), rho(imt,jmt) ,ztp(imt) )
      t(1)=trange(1)
      do i=2,imt
        t(i)=t(i-1)+trange(3)
      enddo
      s(1)=srange(1)
      do i=2,jmt
        s(i)=s(i-1)+srange(3)
      enddo
      ncid = nccre (file, NCCLOB, iret)
      iret=nf_set_fill(ncid, NF_NOFILL, iret)
      tdim  = ncddef(ncid, 'x', imt, iret)
      sdim  = ncddef(ncid, 'y', jmt, iret)
      zdim  = ncddef(ncid, 'depth', km, iret)
      dims(1)  = tdim
      tid  = ncvdef (ncid,'x',NCFLOAT,1,dims,iret)
      dims(1)  = sdim
      sid  = ncvdef (ncid,'y',NCFLOAT,1,dims,iret)
      dims(1)  = zdim
      zid  = ncvdef (ncid,'depth',NCFLOAT,1,dims,iret)
      name = 'Potential temparature'
      call ncaptc(ncid,tid,'long_name',NCCHAR,len_trim(name),name,iret) 
      name = 'deg C'
      call ncaptc(ncid,tid,'units',NCCHAR,len_trim(name),name,iret) 
      name = 'Salinity'
      call ncaptc(ncid,sid,'long_name',NCCHAR,len_trim(name),name,iret) 
      name = 'psu'
      call ncaptc(ncid,sid,'units',NCCHAR,len_trim(name),name,iret) 
      name = 'Depth'
      call ncaptc(ncid,zid,'long_name',NCCHAR,len_trim(name),name,iret) 
      name = 'm'
      call ncaptc(ncid,zid,'units',NCCHAR,len_trim(name),name,iret) 
      dims(1:3)  = (/tdim,sdim,zdim/)
      rid  = ncvdef (ncid,'rho',NCFLOAT,3,dims,iret)
      drdtid  = ncvdef (ncid,'drhodt',NCFLOAT,3,dims,iret)
      drdsid  = ncvdef (ncid,'drhods',NCFLOAT,3,dims,iret)
      name = 'In situ (model) density anomaly'
      call ncaptc(ncid,rid,'long_name',NCCHAR,len_trim(name),name,iret) 
      name = 'kg/m^3'
      call ncaptc(ncid,rid,'units',NCCHAR,len_trim(name),name,iret) 
      name = 'Partial derivative of density to temperatur'
      call ncaptc(ncid,drdtid,'long_name',NCCHAR,
     &            len_trim(name),name,iret) 
      name = 'kg/m^3/K'
      call ncaptc(ncid,drdtid,'units',NCCHAR,len_trim(name),name,iret) 
      name = 'Partial derivative of density to salinity'
      call ncaptc(ncid,drdsid,'long_name',NCCHAR,
     &            len_trim(name),name,iret) 
      name = 'kg/m^3/psu'
      call ncaptc(ncid,drdsid,'units',NCCHAR,len_trim(name),name,iret) 
      call ncendf(ncid, iret)
      allocate(buf(imt,jmt) )
      start=1; count=imt; buf(1:imt,1)=t
      call ncvpt(ncid, tid, start, count,buf(1:imt,1), iret)
      start=1; count=jmt; buf(1,1:jmt)=s
      call ncvpt(ncid, sid, start, count,buf(1,1:jmt), iret)
      start=1; count=km;x4=zt
      call ncvpt(ncid, zid, start, count,x4, iret)
      do k=1,km
       start=(/1,1,k/); count=(/imt,jmt,1/)
      
       ztp = zt(k)*100.
       do j=1,jmt
        ss(:) = s(j)
        call tic('density')
        call model_dens(t,(ss-35)/1000.,rho(:,j),k,imt
#ifdef partial_cell
     &                       ,ztp
#endif
     &                       )
        call toc('density')
       enddo
       buf=rho*1000.
       call ncvpt(ncid, rid, start, count,buf, iret)
       do j=1,jmt
        ss(:) = s(j)
        call tic('drhodt')
        call drhodt(t,(ss-35)/1000.,rho(:,j),k,imt
#ifdef partial_cell
     &                       ,ztp
#endif
     &                       )
        call toc('drhodt')
       enddo
       buf=rho*1000.
       call ncvpt(ncid, drdtid, start, count,buf, iret)
       do j=1,jmt
        ss(:) = s(j)
        call tic('drhods')
        call drhods(t,(ss-35)/1000.,rho(:,j),k,imt
#ifdef partial_cell
     &                       ,ztp
#endif
     &                       )
        call toc('drhods')
       enddo
       buf=rho*1000./1000.
       call ncvpt(ncid, drdsid, start, count,buf, iret)
      enddo
      call ncclos (ncid, iret)
      deallocate( t, s, rho ,ztp , buf )
      print*,' density time summary    = ',timing_secs('density') ,' s'
      print*,' drhodt  time summary    = ',timing_secs('drhodt') ,' s'
      print*,' drhods  time summary    = ',timing_secs('drhods') ,' s'
#endif
      end subroutine diag_eq_of_state


      program test_dens
      use dens_module
      implicit none
      integer :: k
      real :: z_flame(45) 
      data (z_flame(k),k=1,45) / 5.0, 15.0, 25.881020, 37.044724,
     &  49.266403, 61.977428, 75.989777, 90.780373, 107.218124, 
     & 124.851761, 144.641510, 166.253128, 190.798477, 218.141647,
     & 250.656326, 288.517487, 332.659851, 385.779755, 446.269775,
     & 516.055237, 587.043152, 664.011780, 742.207397, 826.629150,
     & 916.737427, 1018.158203, 1134.898682, 1272.809448, 1438.658081,
     & 1641.341919, 1878.658081, 2125.000000, 2375.000000, 2625.0,
     & 2875.000000, 3125.000000, 3375.000000, 3625.000000, 3875.0,
     & 4125.000000, 4375.000000, 4625.000000, 4875.000000, 5125.0,
     & 5375.000000/
      call init_dens_module(z_flame*100.,45,0)
      call diag_eq_of_state('test_dens.cdf',45,z_flame)
      end program test_dens

      subroutine barrier
      end

      subroutine halt_stop(s)
      character*(*) s
      print*,s
      stop
      end
#endif
