
!=======================================================================
!      solve two dimensional Possion equation
!           A * dpsi = forc,  where A = nabla_h^2  
!      with Neumann boundary conditions
!      used for surface pressure or free surface
!=======================================================================


 subroutine congrad2D(nx_,ny_,cf, forc,max_iterations, iterations, epsilon)
      use pyOM_module   
      implicit none
      integer :: nx_,ny_
      integer :: max_iterations,iterations, i,j,n
      real*8 :: epsilon,estimated_error
      logical :: converged, diverged
      real*8 :: forc(nx_,ny_)
      real*8 :: res(nx,ny),Z(nx,ny),Zres(nx,ny)
      real*8 :: ss(nx,ny),As(nx,ny)
      real*8 :: cf(nx_,ny_,3,3)
      real*8 :: zresmax,betakm1,betak,betak_min=0,betaquot,s_dot_As,smax
      real*8 :: alpha,step,step1=0,convergence_rate
      real*8,external :: absmax_2D,dot2_2D

      Z=0.;Zres=0.;ss=0.;As=0.
!-----------------------------------------------------------------------
!     impose boundary conditions on guess
!     dpsi(0) = guess
!-----------------------------------------------------------------------
      call setcyclic2D(nx_,ny_,p_surf)
!-----------------------------------------------------------------------
!     make approximate inverse operator Z (always even symmetry)
!-----------------------------------------------------------------------
      call make_inv_2D (nx_,ny_,cf, Z)
      call setcyclic2D(nx_,ny_,Z)

!-----------------------------------------------------------------------
!     res(0)  = forc - A * eta(0)
!-----------------------------------------------------------------------
      call op9_vec_2D(nx_,ny_,cf, p_surf, res)
      do j=js_pe,je_pe
       do i=1,nx
        res(i,j) = forc(i,j) - res(i,j)
       enddo
      enddo
      call setcyclic2D(nx_,ny_,res)
!-----------------------------------------------------------------------
!     Zres(k-1) = Z * res(k-1)
!     see if guess is a solution, bail out to avoid division by zero
!-----------------------------------------------------------------------
      n = 0
      diverged=.false.
      call inv_op_2D(nx_,ny_,Z, res, Zres)
      call setcyclic2D(nx_,ny_,Zres)
      Zresmax = absmax_2D(nx_,ny_,Zres)
!
!       Assume convergence rate of 0.99 to extrapolate error
!
      if (100.0 * Zresmax .lt. epsilon) then
         estimated_error = 100.0 * Zresmax 
         goto 101
      endif
!-----------------------------------------------------------------------
!     beta(0) = 1
!     ss(0)    = zerovector()
!-----------------------------------------------------------------------
      betakm1 = 1.0
      ss=0.
!-----------------------------------------------------------------------
!     begin iteration loop
!-----------------------------------------------------------------------
      do n = 1,max_iterations
!-----------------------------------------------------------------------
!       Zres(k-1) = Z * res(k-1)
!-----------------------------------------------------------------------
        call inv_op_2D(nx_,ny_,Z, res, Zres)
        call setcyclic2D(nx_,ny_,Zres)
!-----------------------------------------------------------------------
!       beta(k)   = res(k-1) * Zres(k-1)
!-----------------------------------------------------------------------
        betak = dot2_2D(nx_,ny_,Zres, res)

        if (n .eq. 1) then
          betak_min = abs(betak)
        elseif (n .gt. 2) then
          betak_min = min(betak_min, abs(betak))
          if (abs(betak) .gt. 100.0*betak_min) then
           print*,'PE ',my_pe,' : ','WARNING: 2D solver terminated because correction', &
          '         steps are diverging. Probable cause...roundoff'
            diverged=.true.
            goto 101
          endif
        endif
!-----------------------------------------------------------------------
!       ss(k)      = Zres(k-1) + (beta(k)/beta(k-1)) * ss(k-1)
!-----------------------------------------------------------------------
        betaquot = betak/betakm1

        do j=js_pe,je_pe
         do i=1,nx
          ss(i,j) = Zres(i,j) + betaquot * ss(i,j)
         enddo
        enddo
        call setcyclic2D(nx_,ny_,ss)
        call border_exchg2D(nx_,ny_,ss,1)

!-----------------------------------------------------------------------
!       As(k)     = A * ss(k)
!-----------------------------------------------------------------------
        call op9_vec_2D(nx_,ny_,cf, ss, As)
        call setcyclic2D(nx_,ny_,As)
!-----------------------------------------------------------------------
!       If ss=0 then the division for alpha(k) gives a float exception.
!       Assume convergence rate of 0.99 to extrapolate error.
!       Also assume alpha(k) ~ 1.
!-----------------------------------------------------------------------

        s_dot_As = dot2_2D(nx_,ny_,ss, As)

        if (abs(s_dot_As) .lt. abs(betak)*1.e-10) then
          smax = absmax_2D(nx_,ny_,ss)
          estimated_error = 100.0 * smax 
           goto 101
        endif
!-----------------------------------------------------------------------
!       alpha(k)  = beta(k) / (ss(k) * As(k))
!-----------------------------------------------------------------------
        alpha = betak / s_dot_As

!-----------------------------------------------------------------------
!       update values:
!       eta(k)   = eta(k-1) + alpha(k) * ss(k)
!       res(k)    = res(k-1) - alpha(k) * As(k)
!-----------------------------------------------------------------------
        do j=js_pe,je_pe
         do i=1,nx
          p_surf(i,j)  = p_surf(i,j) + alpha * ss(i,j)
          res(i,j)     = res(i,j)   - alpha * As(i,j)
         enddo
        enddo

! adjust this parameter for better performance
! on massive parallel architectures
!        if ((mod(n,10)==0).or.(n==1)) then  
        smax = absmax_2D(nx_,ny_,ss)
!-----------------------------------------------------------------------
!       test for convergence
!       if (estimated_error) < epsilon) exit
!-----------------------------------------------------------------------
        step = abs(alpha) * smax
        if (n .eq. 1) then
          step1 = step
          estimated_error = step
          if (step .lt. epsilon) goto 101
        else if (step .lt. epsilon) then
          convergence_rate = exp(log(step/step1)/(n-1))
          estimated_error = step*convergence_rate/(1.0-convergence_rate)
          if (estimated_error .lt. epsilon) goto 101
        end if
!        endif ! mod(n,10)

        betakm1 = betak
!        print*,'PE=',my_pe,'n=',n,' error=',step,' eps=',epsilon
      end do
!-----------------------------------------------------------------------
!     end of iteration loop
!-----------------------------------------------------------------------
  101 continue
      if ((n .gt. max_iterations).or.(diverged)) then
          print*,'PE ',my_pe,' : 2D-Poisson solver is not converged '
        converged = .false.
      else
        converged = .true.
      end if
      iterations = n
!      if (.not.converged) call halt_stop('  ')
 end subroutine congrad2D



 subroutine op9_vec_2D(nx_,ny_,cf, p1, res)
      use pyOM_module   
      implicit none
!-----------------------------------------------------------------------
!                       res = A *eta 
!-----------------------------------------------------------------------
      integer :: nx_,ny_
      real*8 :: cf(nx_,ny_,3,3) 
      real*8 :: p1(nx_,ny_), res(nx_,ny_)
      integer :: i,j,ii,jj,js,je
      js =max(2,js_pe);   je  = min(je_pe,ny-1)
      res(:,js_pe:je_pe)=0.
      do jj=-1,1
       do ii=-1,1
      do j=js,je
       do i=2,nx-1
        res(i,j) = res(i,j) + cf(i,j,ii+2,jj+2)*p1(i+ii,j+jj) 
       end do
      end do
       end do
      end do
 end subroutine op9_vec_2D


 subroutine inv_op_2D(nx_,ny_,Z, res, Zres)
      use pyOM_module   
      implicit none
!-----------------------------------------------------------------------
!     apply and approximate inverse Z or the operator A
!-----------------------------------------------------------------------
      real*8 ::  Z(nx_,ny_),res(nx_,ny_),Zres(nx_,ny_)
      integer :: i,j
      integer :: nx_,ny_
      do j=js_pe,je_pe
       do i=1,nx
        Zres(i,j) = Z(i,j) * res(i,j)
       end do
      end do
 end subroutine inv_op_2D


 subroutine make_inv_2D (nx_,ny_,cf, Z)
      use pyOM_module   
      implicit none
!-----------------------------------------------------------------------
!     construct an approximate inverse Z to A
!-----------------------------------------------------------------------
      integer :: nx_,ny_
      real*8 :: cf(nx_,ny_,3,3) 
      real*8 ::  Z(nx_,ny_)
      integer :: i,j
      do j=js_pe,je_pe
       do i=1,nx
        if (cf(i,j,2,2) .ne. 0.0) then
         Z(i,j) = 1./cf(i,j,2,2)
        else
         Z(i,j)=0.
        endif
       end do
      end do
 end subroutine make_inv_2D


 real*8 function absmax_2D(nx_,ny_,p1)
      use pyOM_module   
      implicit none
      integer :: nx_,ny_
      real*8 :: p1(nx_,ny_),s2
      integer :: i,j
      s2=0
      do j=js_pe,je_pe
       do i=2,nx-1
        s2 = max( abs(p1(i,j)*maskT(i,j,nz-1)), s2 )
       enddo
      enddo
      if (n_pes>1) call global_max(s2)
      absmax_2D=s2
 end function absmax_2D


 real*8  function dot2_2D(nx_,ny_,p1,p2)
      use pyOM_module   
      implicit none
      integer :: nx_,ny_
      real*8 :: p1(nx_,ny_),p2(nx_,ny_),s2
      integer :: i,j
      s2=0
      do j=js_pe,je_pe
       do i=1,nx
        s2 = s2+p1(i,j)*p2(i,j)*maskt(i,j,nz-1)
       enddo
      enddo
      call global_sum(s2)
      dot2_2D=s2
 end function dot2_2D


