
!=======================================================================
!       time averaging diagnostics
!       linked in the code in driver only
!       Reference buoyancy profile given by N_0 
!=======================================================================


module time_averages_module
!=======================================================================
!        Main module for time averages
!=======================================================================
      implicit none
      integer :: nitts

      logical :: diag_eddy_fluxes           =.false.
      logical :: diag_eddy_flux_tendency    =.false.
      logical :: diag_potential_vorticity   =.false.
      logical :: remove_back_stratification =.false.

      real*8, allocatable :: umean(:,:,:),vmean(:,:,:),wmean(:,:,:)
      real*8, allocatable :: bmean(:,:,:),pmean(:,:,:)
      real*8, allocatable :: bs(:,:,:),ps(:,:,:),b3(:,:,:)
      real*8, allocatable :: uvar(:,:,:),vvar(:,:,:),wvar(:,:,:)
      real*8, allocatable :: bvar(:,:,:), pvar(:,:,:), uv(:,:,:,:)

      real*8, allocatable :: ub(:,:,:,:),vb(:,:,:,:),wb(:,:,:,:)
      real*8, allocatable :: ubb(:,:,:,:),vbb(:,:,:,:),wbb(:,:,:,:)
      real*8, allocatable :: ub3(:,:,:,:),vb3(:,:,:,:),wb3(:,:,:,:)

      real*8, allocatable :: ugeo1(:,:,:),ugeo2(:,:,:)
      real*8, allocatable :: vgeo1(:,:,:),vgeo2(:,:,:)
      real*8, allocatable :: bpx(:,:,:,:),bpy(:,:,:,:)
      real*8, allocatable :: bfv(:,:,:,:),bfu(:,:,:,:)
      real*8, allocatable :: uwN(:,:,:,:),vwN(:,:,:,:)

      integer :: nr_tr  = 0  ! numbers of tracers to be considered
      real*8, allocatable :: tmean(:,:,:,:),tvar(:,:,:,:),t3(:,:,:,:)
      real*8, allocatable :: ut(:,:,:,:,:),vt(:,:,:,:,:),wt(:,:,:,:,:)
      real*8, allocatable :: utt(:,:,:,:,:),vtt(:,:,:,:,:),wtt(:,:,:,:,:)
      real*8, allocatable :: ut3(:,:,:,:,:),vt3(:,:,:,:,:),wt3(:,:,:,:,:)
      real*8, allocatable :: rv(:,:,:),pv(:,:,:)
      real*8, allocatable :: qmean(:,:,:),qvar(:,:,:),q3(:,:,:)
      real*8, allocatable :: uq(:,:,:,:),vq(:,:,:,:),wq(:,:,:,:)
      real*8, allocatable :: uqq(:,:,:,:),vqq(:,:,:,:),wqq(:,:,:,:)
      real*8, allocatable :: uq3(:,:,:,:),vq3(:,:,:,:),wq3(:,:,:,:)
      real*8,parameter :: epsln     = 1.0d-20
      real*8,parameter :: spval = -1.0d33

      real*8, allocatable, dimension(:) :: b_r, p_r
      real*8,parameter :: N_0 = 0.004
end module time_averages_module


subroutine init_time_averages
!=======================================================================
!      Initialize time mean diagnostics
!=======================================================================
      use pyOM_module   
      use time_averages_module
      use fcontrol_module
      implicit none
      logical :: file_exists = .false.
      integer :: n,k,ierr

      ierr=0
      if (enable_diag_tracer) nr_tr = nt
      if (my_pe==0) then
       print*,' Initializing time averaging module '
       print*,' found ',nr_tr,' tracer '
      endif
      allocate( b_r(nz) ); b_r=0.0
      allocate( p_r(nz) ); p_r=0.0
 if (remove_back_stratification) then
      b_r = -zt*N_0**2
      p_r(nz) = 0
      do k=nz-1,1,-1
       p_r(k)=p_r(k+1)+0.5*(b_r(k+1)+b_r(k))*dz
      enddo
 endif
      allocate(bs(nx ,ny ,nz),ps(nx ,ny ,nz) );bs=0.;ps=0.
      allocate(umean(nx ,ny ,nz),vmean(nx ,ny ,nz),wmean(nx ,ny ,nz))
      allocate(bmean(nx ,ny ,nz),pmean(nx ,ny ,nz))
      allocate(uvar(nx ,ny ,nz),vvar(nx ,ny ,nz),wvar(nx ,ny ,nz))
      allocate(bvar(nx ,ny ,nz),pvar(nx ,ny ,nz),uv(nx ,ny ,nz,4))
      allocate(b3(nx ,ny ,nz))
 if (diag_eddy_fluxes) then
      allocate(ub(nx ,ny ,nz,2),vb(nx ,ny ,nz,2))
      allocate(ubb(nx ,ny ,nz,2),vbb(nx ,ny ,nz,2))
      allocate(wb(nx ,ny ,nz,2),wbb(nx ,ny ,nz,2))
      allocate(ub3(nx ,ny ,nz,2),vb3(nx ,ny ,nz,2))
      allocate(wb3(nx ,ny ,nz,2))
 endif
 if (diag_eddy_flux_tendency) then
      allocate(bpx(nx ,ny ,nz,3),bpy(nx ,ny ,nz,3))
      allocate(bfv(nx ,ny ,nz,6),bfu(nx ,ny ,nz,6))
      allocate(ugeo1(nx ,ny ,nz),ugeo2(nx ,ny ,nz) );ugeo1=0.;ugeo2=0.
      allocate(vgeo1(nx ,ny ,nz),vgeo2(nx ,ny ,nz) );vgeo1=0.;vgeo2=0.
      allocate(uwN(nx ,ny ,nz,4),vwN(nx ,ny ,nz,4))
 endif
      allocate(tmean(nx ,ny ,nz,nr_tr),tvar(nx ,ny ,nz,nr_tr))
      allocate(t3(nx ,ny ,nz,nr_tr) )
 if (diag_eddy_fluxes) then
      allocate(ut(nx ,ny ,nz,2,nr_tr),vt(nx ,ny ,nz,2,nr_tr))
      allocate(utt(nx ,ny ,nz,2,nr_tr),vtt(nx ,ny ,nz,2,nr_tr))
      allocate(wt(nx ,ny ,nz,2,nr_tr),wtt(nx ,ny ,nz,2,nr_tr))
      allocate(ut3(nx ,ny ,nz,2,nr_tr),vt3(nx ,ny ,nz,2,nr_tr))
      allocate(wt3(nx ,ny ,nz,2,nr_tr))
 endif
 if (diag_potential_vorticity) then
      allocate(rv(nx ,ny ,nz),pv(nx ,ny ,nz))
      allocate(qmean(nx ,ny ,nz),qvar(nx ,ny ,nz),q3(nx ,ny ,nz))
      allocate(uq(nx ,ny ,nz,2),vq(nx ,ny ,nz,2))
      allocate(uqq(nx ,ny ,nz,2),vqq(nx ,ny ,nz,2))
      allocate(wq(nx ,ny ,nz,2),wqq(nx ,ny ,nz,2))
      allocate(uq3(nx ,ny ,nz,2),vq3(nx ,ny ,nz,2))
      allocate(wq3(nx ,ny ,nz,2))
 endif
      inquire( FILE='restart_ave.dta', EXIST=file_exists )
      if (file_exists) then 
       call read_time_averages_restart(ierr)
      else
       if (my_pe==0) print*,' cannot find restart file '
       nitts=0
       umean=0;vmean=0;wmean=0;bmean=0;pmean=0
       uvar=0;vvar=0;wvar=0;bvar=0;pvar=0; uv=0.;b3=0
 if (diag_eddy_fluxes) then
       ub=0;vb=0;wb=0; ubb=0;vbb=0;wbb=0
       ub3=0;vb3=0;wb3=0.
 endif
 if (diag_eddy_flux_tendency) then
       bpx=0;bpy=0;bfv=0;bfu=0;uwN=0;vwN=0
 endif
       do n=1,nr_tr
        tmean(:,:,:,n)=0;tvar(:,:,:,n)=0;t3(:,:,:,n)=0
 if (diag_eddy_fluxes) then
        ut(:,:,:,:,n)=0;vt(:,:,:,:,n)=0;wt(:,:,:,:,n)=0; 
        utt(:,:,:,:,n)=0;vtt(:,:,:,:,n)=0;wtt(:,:,:,:,n)=0
        ut3(:,:,:,:,n)=0;vt3(:,:,:,:,n)=0;wt3(:,:,:,:,n)=0
 endif
       enddo
 if (diag_potential_vorticity) then
       qmean=0;qvar=0;q3=0
       uq=0;vq=0;wq=0;wqq=0;uqq=0;vqq=0
       vq3=0;uq3=0;wq3=0
 endif
      endif
      if (my_pe==0) print*,' Done initializing time averaging module'
end subroutine init_time_averages


subroutine read_time_averages_restart(ierr)
!=======================================================================
!       read restart
!=======================================================================
      use pyOM_module   
      use time_averages_module
      implicit none
      integer, intent(out) :: ierr
      integer :: io,n,j,k,nx_,ny_,nz_

       ierr=0
       if (my_pe==0) then 
         print*,' reading data from restart file '
         call get_free_iounit(io,ierr)
         if (ierr/=0) goto 10
         open(io,file='restart_ave.dta',form='unformatted',status='old',err=10)
         read(io,err=10) nx_,ny_,nz_
         if (nx/=nx_ .or. ny/=ny_ .or. nz/= nz_) then 
          print*,' read from restart dimensions: ',nx_,ny_,nz_
          print*,' does not match dimensions   : ',nx,ny,nz
          goto 10
         endif
         read(io,err=10) nitts 
         print*,' number of itts = ',nitts
         do j=1,ny 
          read(io,err=10) umean(:,j,:), vmean(:,j,:), wmean(:,j,:)
          read(io,err=10) bmean(:,j,:), pmean(:,j,:)
          read(io,err=10) uvar(:,j,:), vvar(:,j,:), wvar(:,j,:)
          read(io,err=10) bvar(:,j,:), pvar(:,j,:),uv(:,j,:,:),b3(:,j,:)
 if (diag_eddy_fluxes) then
          read(io,err=10) ub(:,j,:,:), vb(:,j,:,:), wb(:,j,:,:)
          read(io,err=10) ubb(:,j,:,:),vbb(:,j,:,:),wbb(:,j,:,:)
          read(io,err=10) ub3(:,j,:,:),vb3(:,j,:,:),wb3(:,j,:,:)
 endif
 if (diag_eddy_flux_tendency) then
          read(io,err=10) bpx(:,j,:,:),bpy(:,j,:,:)
          read(io,err=10) bfv(:,j,:,:),bfu(:,j,:,:)
          read(io,err=10) uwN(:,j,:,:),vwN(:,j,:,:)
 endif
          do n=1,nr_tr
           read(io,err=10) tmean(:,j,:,n), tvar(:,j,:,n),t3(:,j,:,n)
 if (diag_eddy_fluxes) then
           read(io,err=10) ut(:,j,:,:,n), vt(:,j,:,:,n), wt(:,j,:,:,n)
           read(io,err=10) utt(:,j,:,:,n),vtt(:,j,:,:,n),wtt(:,j,:,:,n)
           read(io,err=10) ut3(:,j,:,:,n),vt3(:,j,:,:,n),wt3(:,j,:,:,n)
 endif
         enddo
 if (diag_potential_vorticity) then
          read(io,err=10) qmean(:,j,:), qvar(:,j,:),q3(:,j,:)
          read(io,err=10) uq(:,j,:,:), vq(:,j,:,:), wq(:,j,:,:)
          read(io,err=10) uqq(:,j,:,:),vqq(:,j,:,:),wqq(:,j,:,:)
          read(io,err=10) uq3(:,j,:,:),vq3(:,j,:,:),wq3(:,j,:,:)
 endif
         enddo
         goto 20
 10      ierr=-1
         print*,' error while reading restart'
 20      print*,' done reading restart'
         close(io)
       endif

      call pe0_send_int(ierr)
      if (ierr/=0) return

      call pe0_send_int(nitts)
      call pe0_send_3d(nx,ny,nz,umean)
      call pe0_send_3d(nx,ny,nz,vmean)
      call pe0_send_3d(nx,ny,nz,wmean)
      call pe0_send_3d(nx,ny,nz,pmean)
      call pe0_send_3d(nx,ny,nz,bmean)
      call pe0_send_3d(nx,ny,nz,uvar)
      call pe0_send_3d(nx,ny,nz,vvar)
      call pe0_send_3d(nx,ny,nz,wvar)
      call pe0_send_3d(nx,ny,nz,bvar)
      call pe0_send_3d(nx,ny,nz,pvar)
      call pe0_send_3d(nx,ny,nz,uv)
      call pe0_send_3d(nx,ny,nz,b3)
 if (diag_eddy_fluxes) then
     do n=1,2
      call pe0_send_3d(nx,ny,nz,ub(:,:,:,n))
      call pe0_send_3d(nx,ny,nz,vb(:,:,:,n))
      call pe0_send_3d(nx,ny,nz,wb(:,:,:,n))
      call pe0_send_3d(nx,ny,nz,ubb(:,:,:,n))
      call pe0_send_3d(nx,ny,nz,vbb(:,:,:,n))
      call pe0_send_3d(nx,ny,nz,wbb(:,:,:,n))
      call pe0_send_3d(nx,ny,nz,ub3(:,:,:,n))
      call pe0_send_3d(nx,ny,nz,vb3(:,:,:,n))
      call pe0_send_3d(nx,ny,nz,wb3(:,:,:,n))
      do k=1,nr_tr
        call pe0_send_3d(nx,ny,nz,ut(:,:,:,n,k))
        call pe0_send_3d(nx,ny,nz,vt(:,:,:,n,k))
        call pe0_send_3d(nx,ny,nz,wt(:,:,:,n,k))
        call pe0_send_3d(nx,ny,nz,utt(:,:,:,n,k))
        call pe0_send_3d(nx,ny,nz,vtt(:,:,:,n,k))
        call pe0_send_3d(nx,ny,nz,wtt(:,:,:,n,k))
        call pe0_send_3d(nx,ny,nz,ut3(:,:,:,n,k))
        call pe0_send_3d(nx,ny,nz,vt3(:,:,:,n,k))
        call pe0_send_3d(nx,ny,nz,wt3(:,:,:,n,k))
      enddo
     enddo
 endif
 if (diag_eddy_flux_tendency) then
     do n=1,3
      call pe0_send_3d(nx,ny,nz,bpx(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,bpy(:,:,:,n) )
     enddo
     do n=1,6
      call pe0_send_3d(nx,ny,nz,bfv(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,bfu(:,:,:,n) )
     enddo
     do n=1,4
      call pe0_send_3d(nx,ny,nz,uwN(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,vwN(:,:,:,n) )
     enddo
 endif
 if (diag_potential_vorticity) then
      call pe0_send_3d(nx,ny,nz,qmean)
      call pe0_send_3d(nx,ny,nz,qvar)
      call pe0_send_3d(nx,ny,nz,q3)
     do n=1,2
      call pe0_send_3d(nx,ny,nz,uq(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,vq(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,wq(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,uqq(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,vqq(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,wqq(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,uq3(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,vq3(:,:,:,n) )
      call pe0_send_3d(nx,ny,nz,wq3(:,:,:,n) )
     enddo
 endif
end subroutine read_time_averages_restart



subroutine write_time_averages_restart(ierr)
!=======================================================================
!       write restart
!=======================================================================
      use pyOM_module   
      use time_averages_module
      implicit none
      integer, intent(out) :: ierr
      integer :: io,n,j,k

      ierr=0
      call pe0_recv_3d(nx,ny,nz,umean)
      call pe0_recv_3d(nx,ny,nz,vmean)
      call pe0_recv_3d(nx,ny,nz,wmean)
      call pe0_recv_3d(nx,ny,nz,pmean)
      call pe0_recv_3d(nx,ny,nz,bmean)
      call pe0_recv_3d(nx,ny,nz,uvar)
      call pe0_recv_3d(nx,ny,nz,vvar)
      call pe0_recv_3d(nx,ny,nz,wvar)
      call pe0_recv_3d(nx,ny,nz,bvar)
      call pe0_recv_3d(nx,ny,nz,pvar)
      call pe0_recv_3d(nx,ny,nz,uv)
      call pe0_recv_3d(nx,ny,nz,b3)
 if (diag_eddy_fluxes) then
     do n=1,2
      call pe0_recv_3d(nx,ny,nz,ub(:,:,:,n))
      call pe0_recv_3d(nx,ny,nz,vb(:,:,:,n))
      call pe0_recv_3d(nx,ny,nz,wb(:,:,:,n))
      call pe0_recv_3d(nx,ny,nz,ubb(:,:,:,n))
      call pe0_recv_3d(nx,ny,nz,vbb(:,:,:,n))
      call pe0_recv_3d(nx,ny,nz,wbb(:,:,:,n))
      call pe0_recv_3d(nx,ny,nz,ub3(:,:,:,n))
      call pe0_recv_3d(nx,ny,nz,vb3(:,:,:,n))
      call pe0_recv_3d(nx,ny,nz,wb3(:,:,:,n))
      do k=1,nr_tr
        call pe0_recv_3d(nx,ny,nz,ut(:,:,:,n,k))
        call pe0_recv_3d(nx,ny,nz,vt(:,:,:,n,k))
        call pe0_recv_3d(nx,ny,nz,wt(:,:,:,n,k))
        call pe0_recv_3d(nx,ny,nz,utt(:,:,:,n,k))
        call pe0_recv_3d(nx,ny,nz,vtt(:,:,:,n,k))
        call pe0_recv_3d(nx,ny,nz,wtt(:,:,:,n,k))
        call pe0_recv_3d(nx,ny,nz,ut3(:,:,:,n,k))
        call pe0_recv_3d(nx,ny,nz,vt3(:,:,:,n,k))
        call pe0_recv_3d(nx,ny,nz,wt3(:,:,:,n,k))
      enddo
     enddo
 endif
 if (diag_eddy_flux_tendency) then
     do n=1,3
      call pe0_recv_3d(nx,ny,nz,bpx(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,bpy(:,:,:,n) )
     enddo
     do n=1,6
      call pe0_recv_3d(nx,ny,nz,bfv(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,bfu(:,:,:,n) )
     enddo
     do n=1,4
      call pe0_recv_3d(nx,ny,nz,uwN(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,vwN(:,:,:,n) )
     enddo
 endif
 if (diag_potential_vorticity) then
      call pe0_recv_3d(nx,ny,nz,qmean)
      call pe0_recv_3d(nx,ny,nz,qvar)
      call pe0_recv_3d(nx,ny,nz,q3)
     do n=1,2
      call pe0_recv_3d(nx,ny,nz,uq(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,vq(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,wq(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,uqq(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,vqq(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,wqq(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,uq3(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,vq3(:,:,:,n) )
      call pe0_recv_3d(nx,ny,nz,wq3(:,:,:,n) )
     enddo
 endif

      if (my_pe==0) then
       print*,' writing restart file for time averages'
       call get_free_iounit(io,ierr)
       if (ierr/=0) goto 10
       open(io,file='restart_ave.dta', form='unformatted',status='unknown',err=10)
       write(io,err=10) nx,ny,nz
       write(io,err=10) nitts
       do j=1,ny
        write(io,err=10) umean(:,j,:),vmean(:,j,:),wmean(:,j,:)
        write(io,err=10) bmean(:,j,:),pmean(:,j,:)
        write(io,err=10) uvar(:,j,:),vvar(:,j,:),wvar(:,j,:)
        write(io,err=10) bvar(:,j,:),pvar(:,j,:),uv(:,j,:,:),b3(:,j,:)
 if (diag_eddy_fluxes) then
        write(io,err=10)  ub(:,j,:,:), vb(:,j,:,:), wb(:,j,:,:)
        write(io,err=10) ubb(:,j,:,:),vbb(:,j,:,:),wbb(:,j,:,:)
        write(io,err=10) ub3(:,j,:,:),vb3(:,j,:,:),wb3(:,j,:,:)
 endif
 if (diag_eddy_flux_tendency) then
        write(io,err=10) bpx(:,j,:,:),bpy(:,j,:,:)
        write(io,err=10) bfv(:,j,:,:),bfu(:,j,:,:)
        write(io,err=10) uwN(:,j,:,:),vwN(:,j,:,:)
 endif
        do k=1,nr_tr
         write(io,err=10) tmean(:,j,:,k),tvar(:,j,:,k),t3(:,j,:,k)
 if (diag_eddy_fluxes) then
         write(io,err=10)  ut(:,j,:,:,k), vt(:,j,:,:,k), wt(:,j,:,:,k)
         write(io,err=10) utt(:,j,:,:,k),vtt(:,j,:,:,k),wtt(:,j,:,:,k)
         write(io,err=10) ut3(:,j,:,:,k),vt3(:,j,:,:,k),wt3(:,j,:,:,k)
 endif
        enddo
 if (diag_potential_vorticity) then
        write(io,err=10)  qmean(:,j,:), qvar(:,j,:), q3(:,j,:)
        write(io,err=10)  uq(:,j,:,:), vq(:,j,:,:), wq(:,j,:,:)
        write(io,err=10) uqq(:,j,:,:),vqq(:,j,:,:),wqq(:,j,:,:)
        write(io,err=10) uq3(:,j,:,:),vq3(:,j,:,:),wq3(:,j,:,:)
 endif
       enddo
       close(io)
       print*,' done '
      endif
      return
 10   ierr=-1
      if (my_pe==0) print*,' error while writing time averages restart '
end subroutine write_time_averages_restart


 subroutine time_averages
!=======================================================================
!        Hauptverteiler function fuer time averages
!=======================================================================
      use pyOM_module   
      use time_averages_module
      implicit none
      integer :: i,j,k,js,je
      real*8 :: fxa
!
!        first calculate auxilliary variables
!
 js=max(2,js_pe); je = min(je_pe,ny -1)

 if  (remove_back_stratification) then
    do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         bs(i,j,k)=b(i,j,k,tau)-b_r(k)
         ps(i,j,k)=p_full(i,j,k,tau)-p_r(k)
        enddo
       enddo
    enddo
 else
    do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         bs(i,j,k)=b(i,j,k,tau)
         ps(i,j,k)=p_full(i,j,k,tau)
        enddo
       enddo
    enddo

    if (enable_back_state) then
       do k=2,nz-1
        do j=js,je 
         do i=2,nx -1
          bs(i,j,k)=bs(i,j,k)+back(i,j,k,tau)
         enddo
        enddo
       enddo
    endif
 endif

 call border_exchg3D(nx,ny,nz,bs,1); call setcyclic3D(nx,ny,nz,bs)
 call border_exchg3D(nx,ny,nz,ps,1); call setcyclic3D(nx,ny,nz,ps)

 if (diag_potential_vorticity) then
!
!       relative potential vorticity
!
      rv=0.0
      do j=js,je 
       do k=2,nz-1
        do i=2,nx -1
          rv(i,j,k)=  (v(i+1,j,k,tau)-v(i,j,k,tau) )/dx*maskU(i,j,k)  &
            -(u(i,j+1,k,tau)-u(i,j,k,tau) )/dx*maskV(i,j,k)
        enddo 
       enddo 
      enddo 
      call border_exchg3D(nx,ny,nz,rv,1); call setcyclic3D(nx,ny,nz,rv)
!
!       Ertel potential vorticity  Q N0^2 = b_z (v_x-u_y + f) + u_z b_y - v_z b_x
!
      pv=0.
      do j=js,je 
       do k=2,nz-1
        do i=2,nx -1
         fxa= (rv(i,j,k)+rv(i-1,j,k)+rv(i,j-1,k)+rv(i-1,j-1,k))/4.0
         pv(i,j,k)=(fxa+coriolis_t(j)  )*  ((b(i,j,k+1,tau)-b(i,j,k,tau))/dz*maskW(i,j,k)  &
          +(b(i,j,k,tau)-b(i,j,k-1,tau))/dz*maskW(i,j,k-1))/  (maskW(i,j,k)+maskW(i,j,k-1)+epsln) 
         pv(i,j,k)=-N_0**(-2)*pv(i,j,k)
        enddo 
       enddo 
      enddo 
      if (enable_back_state) then
       do j=js,je 
        do k=2,nz-1
         do i=2,nx -1
          fxa= (rv(i,j,k)+rv(i-1,j,k)+rv(i,j-1,k)+rv(i-1,j-1,k))/4.0
          pv(i,j,k)=-pv(i,j,k)*N_0**2
          pv(i,j,k)=pv(i,j,k)+ &
            (fxa+coriolis_t(j)  )*  ((back(i,j,k+1,tau)-back(i,j,k,tau))/dz*maskW(i,j,k)  &
           +(back(i,j,k,tau)-back(i,j,k-1,tau))/dz*maskW(i,j,k-1))/  (maskW(i,j,k)+maskW(i,j,k-1)+epsln) 
          pv(i,j,k)=-N_0**(-2)*pv(i,j,k)
         enddo 
        enddo 
       enddo 
      endif
      call border_exchg3D(nx,ny,nz,pv,1); call setcyclic3D(nx,ny,nz,pv)
 endif
!
!       average
!
      nitts=nitts+1      
      call time_averages_means
      call time_averages_eddy_fluxes
      call time_averages_flux_tendency
!
!        write results and restart file
!
 end subroutine time_averages


 subroutine time_averages_means
!=======================================================================
!       sum up the mean variables
!=======================================================================
      use pyOM_module   
      use time_averages_module
      use fcontrol_module
      implicit none
      integer :: i,j,k,js,je
      js=max(2,js_pe); je = min(je_pe,ny -1)
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         umean(i,j,k)=umean(i,j,k)+u(i,j,k,tau)
         vmean(i,j,k)=vmean(i,j,k)+v(i,j,k,tau)
         wmean(i,j,k)=wmean(i,j,k)+w(i,j,k,tau)
         bmean(i,j,k)=bmean(i,j,k)+bs(i,j,k)
         pmean(i,j,k)=pmean(i,j,k)+ps(i,j,k)
         uvar(i,j,k) =uvar(i,j,k) +u(i,j,k,tau)**2
         vvar(i,j,k) =vvar(i,j,k) +v(i,j,k,tau)**2
         wvar(i,j,k) =wvar(i,j,k) +w(i,j,k,tau)**2
         bvar(i,j,k) =bvar(i,j,k) +bs(i,j,k)**2
         b3(i,j,k)    =b3(i,j,k) +bs(i,j,k)**3
         pvar(i,j,k) =pvar(i,j,k) +ps(i,j,k)**2
         uv(i,j,k,1) =uv(i,j,k,1) +u(i  ,j,k,tau)*v(i,j  ,k,tau)
         uv(i,j,k,2) =uv(i,j,k,2) +u(i-1,j,k,tau)*v(i,j  ,k,tau)
         uv(i,j,k,3) =uv(i,j,k,3) +u(i  ,j,k,tau)*v(i,j-1,k,tau)
         uv(i,j,k,4) =uv(i,j,k,4) +u(i-1,j,k,tau)*v(i,j-1,k,tau)
        enddo
       enddo
      enddo
 if (enable_diag_tracer) then
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         tmean(i,j,k,:)=tmean(i,j,k,:)+tr(i,j,k,tau,:)
         tvar(i,j,k,:)=tvar(i,j,k,:)+tr(i,j,k,tau,:)**2
         t3(i,j,k,:)=t3(i,j,k,:)+tr(i,j,k,tau,:)**3
        enddo
       enddo
      enddo
 endif
 if (diag_potential_vorticity) then
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         qmean(i,j,k)=qmean(i,j,k)+pv(i,j,k)
         qvar(i,j,k)=qvar(i,j,k)+pv(i,j,k)**2
         q3(i,j,k)=q3(i,j,k)+pv(i,j,k)**3
        enddo
       enddo
      enddo
 endif
end subroutine time_averages_means


subroutine time_averages_eddy_fluxes
!=======================================================================
!       sum up the eddy and variance fluxes 
!=======================================================================
      use pyOM_module   
      use time_averages_module
      use fcontrol_module
      implicit none
      integer :: i,j,k,js,je

 if (diag_eddy_fluxes) then
      js=max(2,js_pe); je = min(je_pe,ny -1)
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
!        buoyancy eddy fluxes
          ub(i,j,k,1) =ub(i,j,k,1) +u(i,j,k,tau)*bs(i  ,j,k)          
          ub(i,j,k,2) =ub(i,j,k,2) +u(i,j,k,tau)*bs(i+1,j,k)          
          vb(i,j,k,1) =vb(i,j,k,1) +v(i,j,k,tau)*bs(i,j  ,k)          
          vb(i,j,k,2) =vb(i,j,k,2) +v(i,j,k,tau)*bs(i,j+1,k)          
          wb(i,j,k,1) =wb(i,j,k,1) +w(i,j,k,tau)*bs(i,j,k  )          
          wb(i,j,k,2) =wb(i,j,k,2) +w(i,j,k,tau)*bs(i,j,k+1)          
!        buoyancy variance fluxes
          ubb(i,j,k,1)=ubb(i,j,k,1)+u(i,j,k,tau)*bs(i  ,j,k)**2
          ubb(i,j,k,2)=ubb(i,j,k,2)+u(i,j,k,tau)*bs(i+1,j,k)**2
          vbb(i,j,k,1)=vbb(i,j,k,1)+v(i,j,k,tau)*bs(i,j  ,k)**2
          vbb(i,j,k,2)=vbb(i,j,k,2)+v(i,j,k,tau)*bs(i,j+1,k)**2
          wbb(i,j,k,1)=wbb(i,j,k,1)+w(i,j,k,tau)*bs(i,j,k  )**2
          wbb(i,j,k,2)=wbb(i,j,k,2)+w(i,j,k,tau)*bs(i,j,k+1)**2
!        third order fluxes
          ub3(i,j,k,1)=ub3(i,j,k,1)+u(i,j,k,tau)*bs(i  ,j,k)**3
          ub3(i,j,k,2)=ub3(i,j,k,2)+u(i,j,k,tau)*bs(i+1,j,k)**3
          vb3(i,j,k,1)=vb3(i,j,k,1)+v(i,j,k,tau)*bs(i,j  ,k)**3
          vb3(i,j,k,2)=vb3(i,j,k,2)+v(i,j,k,tau)*bs(i,j+1,k)**3
          wb3(i,j,k,1)=wb3(i,j,k,1)+w(i,j,k,tau)*bs(i,j,k  )**3
          wb3(i,j,k,2)=wb3(i,j,k,2)+w(i,j,k,tau)*bs(i,j,k+1)**3
        enddo
       enddo
      enddo
 if (enable_diag_tracer) then
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
!        tracer eddy fluxes
          ut(i,j,k,1,:) =ut(i,j,k,1,:) +u(i,j,k,tau)*tr(i  ,j,k,tau,:)          
          ut(i,j,k,2,:) =ut(i,j,k,2,:) +u(i,j,k,tau)*tr(i+1,j,k,tau,:)          
          vt(i,j,k,1,:) =vt(i,j,k,1,:) +v(i,j,k,tau)*tr(i,j  ,k,tau,:)          
          vt(i,j,k,2,:) =vt(i,j,k,2,:) +v(i,j,k,tau)*tr(i,j+1,k,tau,:)          
          wt(i,j,k,1,:) =wt(i,j,k,1,:)+w(i,j,k,tau)*tr(i,j,k  ,tau,:)          
          wt(i,j,k,2,:) =wt(i,j,k,2,:)+w(i,j,k,tau)*tr(i,j,k+1,tau,:)          
!        tracer variance fluxes
          utt(i,j,k,1,:)=utt(i,j,k,1,:)       +u(i,j,k,tau)*tr(i  ,j,k,tau,:)**2
          utt(i,j,k,2,:)=utt(i,j,k,2,:)       +u(i,j,k,tau)*tr(i+1,j,k,tau,:)**2
          vtt(i,j,k,1,:)=vtt(i,j,k,1,:)       +v(i,j,k,tau)*tr(i,j  ,k,tau,:)**2
          vtt(i,j,k,2,:)=vtt(i,j,k,2,:)       +v(i,j,k,tau)*tr(i,j+1,k,tau,:)**2
          wtt(i,j,k,1,:)=wtt(i,j,k,1,:)       +w(i,j,k,tau)*tr(i,j,k  ,tau,:)**2
          wtt(i,j,k,2,:)=wtt(i,j,k,2,:)       +w(i,j,k,tau)*tr(i,j,k+1,tau,:)**2
!        third order fluxes
          ut3(i,j,k,1,:)=ut3(i,j,k,1,:)      +u(i,j,k,tau)*tr(i  ,j,k,tau,:)**3
          ut3(i,j,k,2,:)=ut3(i,j,k,2,:)      +u(i,j,k,tau)*tr(i+1,j,k,tau,:)**3
          vt3(i,j,k,1,:)=vt3(i,j,k,1,:)      +v(i,j,k,tau)*tr(i,j  ,k,tau,:)**3
          vt3(i,j,k,2,:)=vt3(i,j,k,2,:)      +v(i,j,k,tau)*tr(i,j+1,k,tau,:)**3
          wt3(i,j,k,1,:)=wt3(i,j,k,1,:)      +w(i,j,k,tau)*tr(i,j,k  ,tau,:)**3
          wt3(i,j,k,2,:)=wt3(i,j,k,2,:)      +w(i,j,k,tau)*tr(i,j,k+1,tau,:)**3
        enddo
       enddo
      enddo
 endif
 if (diag_potential_vorticity) then
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
!        pv eddy fluxes
          uq(i,j,k,1) =uq(i,j,k,1) +u(i,j,k,tau)*pv(i  ,j,k)          
          uq(i,j,k,2) =uq(i,j,k,2) +u(i,j,k,tau)*pv(i+1,j,k)          
          vq(i,j,k,1) =vq(i,j,k,1) +v(i,j,k,tau)*pv(i,j  ,k)          
          vq(i,j,k,2) =vq(i,j,k,2) +v(i,j,k,tau)*pv(i,j+1,k)          
          wq(i,j,k,1) =wq(i,j,k,1) +w(i,j,k,tau)*pv(i,j,k  )          
          wq(i,j,k,2) =wq(i,j,k,2) +w(i,j,k,tau)*pv(i,j,k+1)          
!        pv variance fluxes
          uqq(i,j,k,1)=uqq(i,j,k,1)+u(i,j,k,tau)*pv(i  ,j,k)**2
          uqq(i,j,k,2)=uqq(i,j,k,2)+u(i,j,k,tau)*pv(i+1,j,k)**2
          vqq(i,j,k,1)=vqq(i,j,k,1)+v(i,j,k,tau)*pv(i,j  ,k)**2
          vqq(i,j,k,2)=vqq(i,j,k,2)+v(i,j,k,tau)*pv(i,j+1,k)**2
          wqq(i,j,k,1)=wqq(i,j,k,1)+w(i,j,k,tau)*pv(i,j,k  )**2
          wqq(i,j,k,2)=wqq(i,j,k,2)+w(i,j,k,tau)*pv(i,j,k+1)**2
!        third order fluxes
          uq3(i,j,k,1)=uq3(i,j,k,1)+u(i,j,k,tau)*pv(i  ,j,k)**3
          uq3(i,j,k,2)=uq3(i,j,k,2)+u(i,j,k,tau)*pv(i+1,j,k)**3
          vq3(i,j,k,1)=vq3(i,j,k,1)+v(i,j,k,tau)*pv(i,j  ,k)**3
          vq3(i,j,k,2)=vq3(i,j,k,2)+v(i,j,k,tau)*pv(i,j+1,k)**3
          wq3(i,j,k,1)=wq3(i,j,k,1)+w(i,j,k,tau)*pv(i,j,k  )**3
          wq3(i,j,k,2)=wq3(i,j,k,2)+w(i,j,k,tau)*pv(i,j,k+1)**3
        enddo
       enddo
      enddo
 endif
 endif
end subroutine time_averages_eddy_fluxes



subroutine time_averages_flux_tendency
!=======================================================================
!        flux tendencies
!=======================================================================
      use pyOM_module   
      use time_averages_module
      implicit none
      integer :: i,j,k,js,je
      js=max(2,js_pe); je = min(je_pe,ny -1)

 if (diag_eddy_flux_tendency) then
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         bpx(i,j,k,1)=bpx(i,j,k,1)+bs(i  ,j,k)*ps(i  ,j,k)
         bpx(i,j,k,2)=bpx(i,j,k,2)+bs(i+1,j,k)*ps(i  ,j,k)
         bpx(i,j,k,3)=bpx(i,j,k,3)+bs(i  ,j,k)*ps(i+1,j,k)

         bpy(i,j,k,1)=bpy(i,j,k,1)+bs(i,j  ,k)*ps(i,j  ,k)
         bpy(i,j,k,2)=bpy(i,j,k,2)+bs(i,j+1,k)*ps(i,j  ,k)
         bpy(i,j,k,3)=bpy(i,j,k,3)+bs(i,j  ,k)*ps(i,j+1,k)

         bfv(i,j,k,1)=bfv(i,j,k,1) + bs(i  ,j,k)*v(i  ,j  ,k,tau)
         bfv(i,j,k,2)=bfv(i,j,k,2) + bs(i+1,j,k)*v(i  ,j  ,k,tau)
         bfv(i,j,k,3)=bfv(i,j,k,3) + bs(i  ,j,k)*v(i+1,j  ,k,tau)
         bfv(i,j,k,4)=bfv(i,j,k,4) + bs(i  ,j,k)*v(i  ,j-1,k,tau)
         bfv(i,j,k,5)=bfv(i,j,k,5) + bs(i+1,j,k)*v(i  ,j-1,k,tau)
         bfv(i,j,k,6)=bfv(i,j,k,6) + bs(i  ,j,k)*v(i+1,j-1,k,tau)

         bfu(i,j,k,1)=bfu(i,j,k,1) + bs(i,j  ,k)*u(i  ,j  ,k,tau)
         bfu(i,j,k,2)=bfu(i,j,k,2) + bs(i,j+1,k)*u(i  ,j  ,k,tau)
         bfu(i,j,k,3)=bfu(i,j,k,3) + bs(i,j  ,k)*u(i  ,j+1,k,tau)
         bfu(i,j,k,4)=bfu(i,j,k,4) + bs(i,j  ,k)*u(i-1,j  ,k,tau)
         bfu(i,j,k,5)=bfu(i,j,k,5) + bs(i,j+1,k)*u(i-1,j  ,k,tau)
         bfu(i,j,k,6)=bfu(i,j,k,6) + bs(i,j  ,k)*u(i-1,j+1,k,tau)

         uwN(i,j,k,1)=uwN(i,j,k,1) + u(i,j,k,tau)*w(i  ,j,k  ,tau)
         uwN(i,j,k,2)=uwN(i,j,k,2) + u(i,j,k,tau)*w(i+1,j,k  ,tau)
         uwN(i,j,k,3)=uwN(i,j,k,3) + u(i,j,k,tau)*w(i  ,j,k-1,tau)
         uwN(i,j,k,4)=uwN(i,j,k,4) + u(i,j,k,tau)*w(i+1,j,k-1,tau)

         vwN(i,j,k,1)=vwN(i,j,k,1) + v(i,j,k,tau)*w(i,j  ,k  ,tau)
         vwN(i,j,k,2)=vwN(i,j,k,2) + v(i,j,k,tau)*w(i,j+1,k  ,tau)
         vwN(i,j,k,3)=vwN(i,j,k,3) + v(i,j,k,tau)*w(i,j  ,k-1,tau)
         vwN(i,j,k,4)=vwN(i,j,k,4) + v(i,j,k,tau)*w(i,j+1,k-1,tau)
        enddo
       enddo
      enddo
 endif
end subroutine time_averages_flux_tendency



 subroutine time_averages_calc
!=======================================================================
!        calculate time mean diagnostics
!=======================================================================
      use pyOM_module   
      use time_averages_module
      implicit none
      integer :: i,j,k,js,je,n,m
      js=max(2,js_pe); je = min(je_pe,ny -1)
!
!        calculate means
!
      n=max(1,nitts)
      umean=umean*maskU/n
      vmean=vmean*maskV/n
      wmean=wmean*maskW/n
      bmean=bmean*maskT/n
      pmean=pmean*maskT/n
      call border_exchg3D(nx,ny,nz,bmean,1); call setcyclic3D(nx,ny,nz,bmean)
      call border_exchg3D(nx,ny,nz,pmean,1); call setcyclic3D(nx,ny,nz,pmean)
      call border_exchg3D(nx,ny,nz,umean,1); call setcyclic3D(nx,ny,nz,umean)
      call border_exchg3D(nx,ny,nz,vmean,1); call setcyclic3D(nx,ny,nz,vmean)
      do k=1,nr_tr
       tmean(:,:,:,k)=tmean(:,:,:,k)*maskT/n
       call border_exchg3D(nx,ny,nz,tmean(:,:,:,k),1)
       call setcyclic3D(nx,ny,nz,tmean(:,:,:,k))
      enddo
 if (diag_potential_vorticity) then
      qmean=qmean*maskT/n
      call border_exchg3D(nx,ny,nz,qmean,1); call setcyclic3D(nx,ny,nz,qmean)
 endif
!
!       Variances
!
      uvar=uvar/n - umean**2
      vvar=vvar/n - vmean**2
      wvar=wvar/n - wmean**2
      bvar=bvar/n - bmean**2
      pvar=pvar/n - pmean**2
!  (b+b')(b+b')(b+b')=(b^2 + 2bb'+b'^2)(b+b')=b^3 + 2b^2b'+b'^2 b + b^2b'+2bb'^2+b'^3=b^3 + 3bb'^2+b'^3
      b3=b3/n - bmean**3-3*bmean*bvar
      do k=1,nr_tr
       tvar(:,:,:,k)=tvar(:,:,:,k)/n - tmean(:,:,:,k)**2
       t3(:,:,:,k)=t3(:,:,:,k)/n - tmean(:,:,:,k)**3 -3*tmean(:,:,:,k)*tvar(:,:,:,k)
      enddo
 if (diag_potential_vorticity) then
      qvar=qvar/n - qmean**2
      q3=q3/n - qmean**3-3*qmean*qvar
 endif

      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         uv(i,j,k,1)=uv(i,j,k,1)/n +umean(i  ,j,k)*vmean(i,j  ,k)
         uv(i,j,k,2)=uv(i,j,k,2)/n +umean(i-1,j,k)*vmean(i,j  ,k)
         uv(i,j,k,3)=uv(i,j,k,3)/n +umean(i  ,j,k)*vmean(i,j-1,k)
         uv(i,j,k,4)=uv(i,j,k,4)/n +umean(i-1,j,k)*vmean(i,j-1,k)
         uv(i,j,k,1)=(uv(i,j,k,1)+uv(i,j,k,2)    +uv(i,j,k,3)+uv(i,j,k,4))/4.0
        enddo
       enddo
      enddo

 if (diag_eddy_fluxes) then
!
!       Eddy fluxes
!
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
!       buoyancy
         ub(i,j,k,1)=ub(i,j,k,1)/n-umean(i,j,k)*bmean(i ,j,k)
         ub(i,j,k,2)=ub(i,j,k,2)/n-umean(i,j,k)*bmean(i+1,j,k)
         vb(i,j,k,1)=vb(i,j,k,1)/n-vmean(i,j,k)*bmean(i,j,k)
         vb(i,j,k,2)=vb(i,j,k,2)/n-vmean(i,j,k)*bmean(i,j+1,k)
         wb(i,j,k,1)=wb(i,j,k,1)/n-wmean(i,j,k)*bmean(i,j,k)
         wb(i,j,k,2)=wb(i,j,k,2)/n-wmean(i,j,k)*bmean(i,j,k+1)
        enddo
       enddo
      enddo

      do m=1,nr_tr
       do k=2,nz-1
        do j=js,je 
         do i=2,nx -1
!       tracer
          ut(i,j,k,1,m)=ut(i,j,k,1,m)/n-umean(i,j,k)*tmean(i,j,k,m)
          ut(i,j,k,2,m)=ut(i,j,k,2,m)/n-umean(i,j,k)*tmean(i+1,j,k,m)
          vt(i,j,k,1,m)=vt(i,j,k,1,m)/n-vmean(i,j,k)*tmean(i,j,k,m)
          vt(i,j,k,2,m)=vt(i,j,k,2,m)/n-vmean(i,j,k)*tmean(i,j+1,k,m)
          wt(i,j,k,1,m)=wt(i,j,k,1,m)/n-wmean(i,j,k)*tmean(i,j,k,m)
          wt(i,j,k,2,m)=wt(i,j,k,2,m)/n-wmean(i,j,k)*tmean(i,j,k+1,m)
         enddo
        enddo
       enddo
      enddo
 if (diag_potential_vorticity) then
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
!       pv
         uq(i,j,k,1)=uq(i,j,k,1)/n-umean(i,j,k)*qmean(i ,j,k)
         uq(i,j,k,2)=uq(i,j,k,2)/n-umean(i,j,k)*qmean(i+1,j,k)
         vq(i,j,k,1)=vq(i,j,k,1)/n-vmean(i,j,k)*qmean(i,j,k)
         vq(i,j,k,2)=vq(i,j,k,2)/n-vmean(i,j,k)*qmean(i,j+1,k)
         wq(i,j,k,1)=wq(i,j,k,1)/n-wmean(i,j,k)*qmean(i,j,k)
         wq(i,j,k,2)=wq(i,j,k,2)/n-wmean(i,j,k)*qmean(i,j,k+1)
        enddo
       enddo
      enddo
 endif
!
!       Variance fluxes
!
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
!       buoyancy
         ubb(i,j,k,1)=ubb(i,j,k,1)/n-umean(i,j,k)*bmean(i,j,k)**2     -2*bmean(i  ,j,k)*ub(i,j,k,1)
         ubb(i,j,k,2)=ubb(i,j,k,2)/n-umean(i,j,k)*bmean(i+1,j,k)**2   -2*bmean(i+1,j,k)*ub(i,j,k,2)
         vbb(i,j,k,1)=vbb(i,j,k,1)/n-vmean(i,j,k)*bmean(i,j,k)**2     -2*bmean(i,j  ,k)*vb(i,j,k,1)
         vbb(i,j,k,2)=vbb(i,j,k,2)/n-vmean(i,j,k)*bmean(i,j+1,k)**2   -2*bmean(i,j+1,k)*vb(i,j,k,2)
         wbb(i,j,k,1)=wbb(i,j,k,1)/n-wmean(i,j,k)*bmean(i,j,k)**2     -2*bmean(i,j,k  )*wb(i,j,k,1)
         wbb(i,j,k,2)=wbb(i,j,k,2)/n-wmean(i,j,k)*bmean(i,j,k+1)**2   -2*bmean(i,j,k+1)*wb(i,j,k,2)


!       third order fluxes of buoyancy
        ub3(i,j,k,1)=ub3(i,j,k,1)/n-umean(i,j,k)*bmean(i,j,k)**3   &
                 -3*bmean(i,j,k)*ubb(i,j,k,1)   -3*bmean(i,j,k)**2*ub(i,j,k,1)
        ub3(i,j,k,2)=ub3(i,j,k,2)/n-umean(i,j,k)*bmean(i+1,j,k)**3   &
                 -3*bmean(i+1,j,k)*ubb(i,j,k,2) -3*bmean(i+1,j,k)**2*ub(i,j,k,2)

        vb3(i,j,k,1)=vb3(i,j,k,1)/n-vmean(i,j,k)*bmean(i,j,k)**3   &
                 -3*bmean(i,j,k)*vbb(i,j,k,1)   -3*bmean(i,j,k)**2*vb(i,j,k,1)
        vb3(i,j,k,2)=vb3(i,j,k,2)/n-vmean(i,j,k)*bmean(i,j+1,k)**3   &
                 -3*bmean(i,j+1,k)*vbb(i,j,k,2) -3*bmean(i,j+1,k)**2*vb(i,j,k,2)

        wb3(i,j,k,1)=wb3(i,j,k,1)/n-wmean(i,j,k)*bmean(i,j,k)**3   &
                 -3*bmean(i,j,k)*wbb(i,j,k,1)  -3*bmean(i,j,k)**2*wb(i,j,k,1)
        wb3(i,j,k,2)=wb3(i,j,k,2)/n-wmean(i,j,k)*bmean(i,j,k+1)**3   &
                 -3*bmean(i,j,k+1)*wbb(i,j,k,2)-3*bmean(i,j,k+1)**2*wb(i,j,k,2)

        enddo
       enddo
      enddo

 if (diag_potential_vorticity) then
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
!       pv
         uqq(i,j,k,1)=uqq(i,j,k,1)/n-umean(i,j,k)*qmean(i,j,k)**2  -2*qmean(i  ,j,k)*uq(i,j,k,1)
         uqq(i,j,k,2)=uqq(i,j,k,2)/n-umean(i,j,k)*qmean(i+1,j,k)**2-2*qmean(i+1,j,k)*uq(i,j,k,2)
         vqq(i,j,k,1)=vqq(i,j,k,1)/n-vmean(i,j,k)*qmean(i,j,k)**2  -2*qmean(i,j  ,k)*vq(i,j,k,1)
         vqq(i,j,k,2)=vqq(i,j,k,2)/n-vmean(i,j,k)*qmean(i,j+1,k)**2-2*qmean(i,j+1,k)*vq(i,j,k,2)
         wqq(i,j,k,1)=wqq(i,j,k,1)/n-wmean(i,j,k)*qmean(i,j,k)**2  -2*qmean(i,j,k  )*wq(i,j,k,1)
         wqq(i,j,k,2)=wqq(i,j,k,2)/n-wmean(i,j,k)*qmean(i,j,k+1)**2-2*qmean(i,j,k+1)*wq(i,j,k,2)

!       third order fluxes of pv
        uq3(i,j,k,1)=uq3(i,j,k,1)/n-umean(i,j,k)*qmean(i,j,k)**3   &
                 -3*qmean(i,j,k)*uqq(i,j,k,1)  -3*qmean(i,j,k)**2*uq(i,j,k,1)
        uq3(i,j,k,2)=uq3(i,j,k,2)/n-umean(i,j,k)*qmean(i+1,j,k)**3   &
                 -3*qmean(i+1,j,k)*uqq(i,j,k,2) -3*qmean(i+1,j,k)**2*uq(i,j,k,2)

        vq3(i,j,k,1)=vq3(i,j,k,1)/n-vmean(i,j,k)*qmean(i,j,k)**3   &
                 -3*qmean(i,j,k)*vqq(i,j,k,1)  -3*qmean(i,j,k)**2*vq(i,j,k,1)
        vq3(i,j,k,2)=vq3(i,j,k,2)/n-vmean(i,j,k)*qmean(i,j+1,k)**3   &
                 -3*qmean(i,j+1,k)*vqq(i,j,k,2)-3*qmean(i,j+1,k)**2*vq(i,j,k,2)

        wq3(i,j,k,1)=wq3(i,j,k,1)/n-wmean(i,j,k)*qmean(i,j,k)**3   &
                 -3*qmean(i,j,k)*wqq(i,j,k,1)  -3*qmean(i,j,k)**2*wq(i,j,k,1)
        wq3(i,j,k,2)=wq3(i,j,k,2)/n-wmean(i,j,k)*qmean(i,j,k+1)**3   &
                 -3*qmean(i,j,k+1)*wqq(i,j,k,2)-3*qmean(i,j,k+1)**2*wq(i,j,k,2)
        enddo
       enddo
      enddo
 endif

!       tracer
      do m=1,nr_tr
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         utt(i,j,k,1,m)=utt(i,j,k,1,m)/n -umean(i,j,k)*tmean(i,j,k,m)**2 -2*tmean(i  ,j,k,m)*ut(i,j,k,1,m)
         utt(i,j,k,2,m)=utt(i,j,k,2,m)/n -umean(i,j,k)*tmean(i+1,j,k,m)**2 -2*tmean(i+1,j,k,m)*ut(i,j,k,2,m)
         vtt(i,j,k,1,m)=vtt(i,j,k,1,m)/n -vmean(i,j,k)*tmean(i,j,k,m)**2  -2*tmean(i,j  ,k,m)*vt(i,j,k,1,m)
         vtt(i,j,k,2,m)=vtt(i,j,k,2,m)/n -vmean(i,j,k)*tmean(i,j+1,k,m)**2 -2*tmean(i,j+1,k,m)*vt(i,j,k,2,m)
         wtt(i,j,k,1,m)=wtt(i,j,k,1,m)/n -wmean(i,j,k)*tmean(i,j,k,m)**2 -2*tmean(i,j,k  ,m)*wt(i,j,k,1,m)
         wtt(i,j,k,2,m)=wtt(i,j,k,2,m)/n -wmean(i,j,k)*tmean(i,j,k+1,m)**2-2*tmean(i,j,k+1,m)*wt(i,j,k,2,m)
!       third order fluxes of tracer
        ut3(i,j,k,1,m)=ut3(i,j,k,1,m)/n-umean(i,j,k)*tmean(i,j,k,m)**3   &
                 -3*tmean(i,j,k,m)*utt(i,j,k,1,m)     -3*tmean(i,j,k,m)**2*ut(i,j,k,1,m)
        ut3(i,j,k,2,m)=ut3(i,j,k,2,m)/n-umean(i,j,k)*tmean(i+1,j,k,m)**3   &
                 -3*tmean(i+1,j,k,m)*utt(i,j,k,2,m)   -3*tmean(i+1,j,k,m)**2*ut(i,j,k,2,m)

        vt3(i,j,k,1,m)=vt3(i,j,k,1,m)/n-vmean(i,j,k)*tmean(i,j,k,m)**3   &
                 -3*tmean(i,j,k,m)*vtt(i,j,k,1,m)   -3*tmean(i,j,k,m)**2*vt(i,j,k,1,m)
        vt3(i,j,k,2,m)=vt3(i,j,k,2,m)/n-vmean(i,j,k)*tmean(i,j+1,k,m)**3   &
                 -3*tmean(i,j+1,k,m)*vtt(i,j,k,2,m) -3*tmean(i,j+1,k,m)**2*vt(i,j,k,2,m)

        wt3(i,j,k,1,m)=wt3(i,j,k,1,m)/n-wmean(i,j,k)*tmean(i,j,k,m)**3   &
                 -3*tmean(i,j,k,m)*wtt(i,j,k,1,m)   -3*tmean(i,j,k,m)**2*wt(i,j,k,1,m)
        wt3(i,j,k,2,m)=wt3(i,j,k,2,m)/n-wmean(i,j,k)*tmean(i,j,k+1,m)**3   &
                 -3*tmean(i,j,k+1,m)*wtt(i,j,k,2,m)  -3*tmean(i,j,k+1,m)**2*wt(i,j,k,2,m)
        enddo
       enddo
      enddo
      enddo

      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         ub(i,j,k,1)=0.5*(ub(i,j,k,1)+ub(i,j,k,2))
         vb(i,j,k,1)=0.5*(vb(i,j,k,1)+vb(i,j,k,2))
         wb(i,j,k,1)=0.5*(wb(i,j,k,1)+wb(i,j,k,2))
         ubb(i,j,k,1)=0.5*(ubb(i,j,k,1)+ubb(i,j,k,2))/2.0
         vbb(i,j,k,1)=0.5*(vbb(i,j,k,1)+vbb(i,j,k,2))/2.0
         wbb(i,j,k,1)=0.5*(wbb(i,j,k,1)+wbb(i,j,k,2))/2.0
         ub3(i,j,k,1)=0.5*(ub3(i,j,k,1)+ub3(i,j,k,2))/3.0
         vb3(i,j,k,1)=0.5*(vb3(i,j,k,1)+vb3(i,j,k,2))/3.0
         wb3(i,j,k,1)=0.5*(wb3(i,j,k,1)+wb3(i,j,k,2))/3.0
        enddo
       enddo
      enddo
 if (diag_potential_vorticity) then
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         uq(i,j,k,1)=0.5*(uq(i,j,k,1)+uq(i,j,k,2))
         vq(i,j,k,1)=0.5*(vq(i,j,k,1)+vq(i,j,k,2))
         wq(i,j,k,1)=0.5*(wq(i,j,k,1)+wq(i,j,k,2))
         uqq(i,j,k,1)=0.5*(uqq(i,j,k,1)+uqq(i,j,k,2))/2.0
         vqq(i,j,k,1)=0.5*(vqq(i,j,k,1)+vqq(i,j,k,2))/2.0
         wqq(i,j,k,1)=0.5*(wqq(i,j,k,1)+wqq(i,j,k,2))/2.0
         uq3(i,j,k,1)=0.5*(uq3(i,j,k,1)+uq3(i,j,k,2))/3.0
         vq3(i,j,k,1)=0.5*(vq3(i,j,k,1)+vq3(i,j,k,2))/3.0
         wq3(i,j,k,1)=0.5*(wq3(i,j,k,1)+wq3(i,j,k,2))/3.0
        enddo
       enddo
      enddo
 endif

      do m=1,nr_tr
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         ut(i,j,k,1,m)=0.5*(ut(i,j,k,1,m) +ut(i,j,k,2,m))
         vt(i,j,k,1,m)=0.5*(vt(i,j,k,1,m) +vt(i,j,k,2,m))
         wt(i,j,k,1,m)=0.5*(wt(i,j,k,1,m)+wt(i,j,k,2,m))

         utt(i,j,k,1,m)=0.5*(utt(i,j,k,1,m) +utt(i,j,k,2,m))/2.0
         vtt(i,j,k,1,m)=0.5*(vtt(i,j,k,1,m) +vtt(i,j,k,2,m))/2.0
         wtt(i,j,k,1,m)=0.5*(wtt(i,j,k,1,m)+wtt(i,j,k,2,m))/2.0

         ut3(i,j,k,1,m)=0.5*(ut3(i,j,k,1,m) +ut3(i,j,k,2,m))/2.0
         vt3(i,j,k,1,m)=0.5*(vt3(i,j,k,1,m) +vt3(i,j,k,2,m))/2.0
         wt3(i,j,k,1,m)=0.5*(wt3(i,j,k,1,m)+wt3(i,j,k,2,m))/2.0
        enddo
       enddo
      enddo
      enddo
 endif

 if (diag_eddy_flux_tendency) then
!
!       flux tendencies :  geostrophic terms
!
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
          bpx(i,j,k,1)=bpx(i,j,k,1)/n-bmean(i  ,j,k)*pmean(i  ,j,k)
          bpx(i,j,k,2)=bpx(i,j,k,2)/n-bmean(i+1,j,k)*pmean(i  ,j,k)
          bpx(i,j,k,3)=bpx(i,j,k,3)/n-bmean(i  ,j,k)*pmean(i+1,j,k)

          bpy(i,j,k,1)=bpy(i,j,k,1)/n-bmean(i,j  ,k)*pmean(i,j  ,k)
          bpy(i,j,k,2)=bpy(i,j,k,2)/n-bmean(i,j+1,k)*pmean(i,j  ,k)
          bpy(i,j,k,3)=bpy(i,j,k,3)/n-bmean(i,j  ,k)*pmean(i,j+1,k)

          bfv(i,j,k,1)=bfv(i,j,k,1)/n-bmean(i  ,j,k)*vmean(i  ,j  ,k)
          bfv(i,j,k,2)=bfv(i,j,k,2)/n-bmean(i+1,j,k)*vmean(i  ,j  ,k)
          bfv(i,j,k,3)=bfv(i,j,k,3)/n-bmean(i  ,j,k)*vmean(i+1,j  ,k)
          bfv(i,j,k,4)=bfv(i,j,k,4)/n-bmean(i  ,j,k)*vmean(i  ,j-1,k)
          bfv(i,j,k,5)=bfv(i,j,k,5)/n-bmean(i+1,j,k)*vmean(i  ,j-1,k)
          bfv(i,j,k,6)=bfv(i,j,k,6)/n-bmean(i  ,j,k)*vmean(i+1,j-1,k)

          bfu(i,j,k,1)=bfu(i,j,k,1)/n-bmean(i,j  ,k)*umean(i  ,j  ,k)
          bfu(i,j,k,2)=bfu(i,j,k,2)/n-bmean(i,j+1,k)*umean(i  ,j  ,k)
          bfu(i,j,k,3)=bfu(i,j,k,3)/n-bmean(i,j  ,k)*umean(i  ,j+1,k)
          bfu(i,j,k,4)=bfu(i,j,k,4)/n-bmean(i,j  ,k)*umean(i-1,j  ,k)
          bfu(i,j,k,5)=bfu(i,j,k,5)/n-bmean(i,j+1,k)*umean(i-1,j  ,k)
          bfu(i,j,k,6)=bfu(i,j,k,6)/n-bmean(i,j  ,k)*umean(i-1,j+1,k)
        enddo
       enddo
      enddo
      call border_exchg3D(nx,ny,nz,bpx(:,:,:,1),1)
      call setcyclic3D(nx,ny,nz,bpx(:,:,:,1))
      call border_exchg3D(nx,ny,nz,bpy(:,:,:,1),1)
      call setcyclic3D(nx,ny,nz,bpy(:,:,:,1))
      call border_exchg3D(nx,ny,nz,bfv(:,:,:,1),1)
      call setcyclic3D(nx,ny,nz,bfv(:,:,:,1))
      call border_exchg3D(nx,ny,nz,bfv(:,:,:,4),1)
      call setcyclic3D(nx,ny,nz,bfv(:,:,:,4))
      call border_exchg3D(nx,ny,nz,bfu(:,:,:,1),1)
      call setcyclic3D(nx,ny,nz,bfu(:,:,:,1))
      call border_exchg3D(nx,ny,nz,bfu(:,:,:,4),1)
      call setcyclic3D(nx,ny,nz,bfu(:,:,:,4))

      ugeo1=0;ugeo2=0; vgeo1=0;vgeo2=0
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
         ugeo1(i,j,k)=-(bpx(i+1,j,k,1)-bpx(i,j,k,2)      +bpx(i,j,k,3)-bpx(i,j,k,1))/(2*dx)
         vgeo1(i,j,k)=-(bpy(i,j+1,k,1)-bpy(i,j,k,2)      +bpy(i,j,k,3)-bpy(i,j,k,1))/(2*dx)
         ugeo2(i,j,k)=(bfv(i,j,k,1)+bfv(i,j,k,2)     +bfv(i,j,k,3)+bfv(i+1,j,k,1)   &
                      +bfv(i,j,k,4)+bfv(i,j,k,5)     +bfv(i,j,k,6)+bfv(i+1,j,k,4) )/8.*coriolis_t(j)
         vgeo2(i,j,k)=-(bfu(i,j,k,1)+bfu(i,j,k,2)   +bfu(i,j,k,4)+bfu(i,j,k,5) )/8.*coriolis_t(j)  &
                      -(bfu(i,j+1,k,1)+bfu(i,j+1,k,4)  +bfu(i,j,k,6)+bfu(i,j,k,3))/8.*coriolis_t(j+1)
        enddo
       enddo
      enddo
!
!       flux tendencies:   vertical advection
!
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
          uwN(i,j,k,1)=uwN(i,j,k,1)/n-umean(i,j,k)*wmean(i  ,j,k)
          uwN(i,j,k,2)=uwN(i,j,k,2)/n-umean(i,j,k)*wmean(i+1,j,k)
          uwN(i,j,k,3)=uwN(i,j,k,3)/n-umean(i,j,k)*wmean(i  ,j,k-1)
          uwN(i,j,k,4)=uwN(i,j,k,4)/n-umean(i,j,k)*wmean(i+1,j,k-1)
          vwN(i,j,k,1)=vwN(i,j,k,1)/n-vmean(i,j,k)*wmean(i,j  ,k)
          vwN(i,j,k,2)=vwN(i,j,k,2)/n-vmean(i,j,k)*wmean(i,j+1,k)
          vwN(i,j,k,3)=vwN(i,j,k,3)/n-vmean(i,j,k)*wmean(i,j  ,k-1)
          vwN(i,j,k,4)=vwN(i,j,k,4)/n-vmean(i,j,k)*wmean(i,j+1,k-1)
        enddo
       enddo
      enddo

   if  (remove_back_stratification) then

      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
          uwN(i,j,k,1)=  uwN(i,j,k,1)*(bmean(i,j,k+1)+bmean(i,j,k))/(4*dz)   &
             +uwN(i,j,k,2)*(bmean(i+1,j,k+1)+bmean(i+1,j,k))/(4*dz)   &
             -uwN(i,j,k,3)*(bmean(i,j,k)+bmean(i,j,k-1))/(4*dz)   &
             -uwN(i,j,k,4)*(bmean(i+1,j,k)+bmean(i+1,j,k-1))/(4*dz)   &
             +uwN(i,j,k,1)*(b_r(k+1)+b_r(k))/(4*dz)   +uwN(i,j,k,2)*(b_r(k+1)+b_r(k))/(4*dz)   &
             -uwN(i,j,k,3)*(b_r(k)+b_r(k-1))/(4*dz)   -uwN(i,j,k,4)*(b_r(k)+b_r(k-1))/(4*dz)
          vwN(i,j,k,1)=  vwN(i,j,k,1)*(bmean(i,j,k+1)+bmean(i,j,k))/(4*dz)  &
             +vwN(i,j,k,2)*(bmean(i,j+1,k+1)+bmean(i,j+1,k))/(4*dz)  &
             -vwN(i,j,k,3)*(bmean(i,j,k)+bmean(i,j,k-1))/(4*dz)  &
             -vwN(i,j,k,4)*(bmean(i,j+1,k)+bmean(i,j+1,k-1))/(4*dz)  &
             +vwN(i,j,k,1)*(b_r(k+1)+b_r(k))/(4*dz)  +vwN(i,j,k,2)*(b_r(k+1)+b_r(k))/(4*dz)  &
             -vwN(i,j,k,3)*(b_r(k)+b_r(k-1))/(4*dz)  -vwN(i,j,k,4)*(b_r(k)+b_r(k-1))/(4*dz)
        enddo
       enddo
      enddo
   else
      do k=2,nz-1
       do j=js,je 
        do i=2,nx -1
          uwN(i,j,k,1)= uwN(i,j,k,1)*(bmean(i,j,k+1)+bmean(i,j,k))/(4*dz)   &
             +uwN(i,j,k,2)*(bmean(i+1,j,k+1)+bmean(i+1,j,k))/(4*dz)   &
             -uwN(i,j,k,3)*(bmean(i,j,k)+bmean(i,j,k-1))/(4*dz)   &
             -uwN(i,j,k,4)*(bmean(i+1,j,k)+bmean(i+1,j,k-1))/(4*dz)
          vwN(i,j,k,1)= vwN(i,j,k,1)*(bmean(i,j,k+1)+bmean(i,j,k))/(4*dz)   &
             +vwN(i,j,k,2)*(bmean(i,j+1,k+1)+bmean(i,j+1,k))/(4*dz)   &
             -vwN(i,j,k,3)*(bmean(i,j,k)+bmean(i,j,k-1))/(4*dz)   &
             -vwN(i,j,k,4)*(bmean(i,j+1,k)+bmean(i,j+1,k-1))/(4*dz)
        enddo
       enddo
      enddo
   endif
 endif
!
!       apply  land mask
!
      where (maskU==0) 
        umean=spval; uvar=spval
      end where
      where (maskV==0) 
        vmean=spval; wvar=spval
      end where
      where (maskW==0) 
        wmean=spval; wvar=spval
      end where
      where (maskT==0) 
       bmean=spval;pmean=spval;b3=spval
       bvar=spval;pvar=spval; uv(:,:,:,1)=spval
      end where
 if (diag_eddy_fluxes) then
      where (maskU==0) 
        ub(:,:,:,1)=spval; ubb(:,:,:,1)=spval;ub3(:,:,:,1)=spval
      end where
      where (maskV==0) 
        vb(:,:,:,1)=spval; vbb(:,:,:,1)=spval;vb3(:,:,:,1)=spval
      end where
      where (maskW==0) 
        wb(:,:,:,1)=spval; wbb(:,:,:,1)=spval;wb3(:,:,:,1)=spval
      end where
 endif
 if (diag_potential_vorticity) then
      where (maskT==0) 
       qmean=spval;q3=spval; qvar=spval;
      end where
      where (maskU==0) 
        uq(:,:,:,1)=spval; uqq(:,:,:,1)=spval;uq3(:,:,:,1)=spval
      end where
      where (maskV==0) 
        vq(:,:,:,1)=spval; vqq(:,:,:,1)=spval;vq3(:,:,:,1)=spval
      end where
      where (maskW==0) 
        wq(:,:,:,1)=spval; wqq(:,:,:,1)=spval;wq3(:,:,:,1)=spval
      end where
 endif

 if (diag_eddy_flux_tendency) then
      where (maskU==0) 
        ugeo1=spval; ugeo2=spval; uwN(:,:,:,1)=spval
      end where
      where (maskV==0) 
        vgeo1=spval; vgeo2=spval; vwN(:,:,:,1)=spval
      end where
 endif
      do k=1,nr_tr
       where (maskT==0) 
        tmean(:,:,:,k)=spval;tvar(:,:,:,k)=spval;t3(:,:,:,k)=spval
       end where
 if (diag_eddy_fluxes) then
       where (maskU==0) 
         ut(:,:,:,1,k)=spval; utt(:,:,:,1,k)=spval;ut3(:,:,:,1,k)=spval
       end where
       where (maskV==0) 
         vt(:,:,:,1,k)=spval; vtt(:,:,:,1,k)=spval;vt3(:,:,:,1,k)=spval
       end where
       where (maskW==0) 
         wt(:,:,:,1,k)=spval; wtt(:,:,:,1,k)=spval;wt3(:,:,:,1,k)=spval
       end where
 endif
      enddo
end subroutine time_averages_calc



subroutine time_averages_write
!=======================================================================
!        write time mean diagnostics
!=======================================================================
      use pyOM_module   
      use time_averages_module
      use fcontrol_module
      implicit none
      include "netcdf.inc"
      integer :: ncid,iret,n,npe, corner(4), edges(4)
      integer :: tdim,ilen,timeid,varid,dims(4)
      integer :: xtdim,xudim,ytdim,yudim,ztdim,zudim
      real*8 :: fxa,time
      character :: name*64, unit*32
!
!        define netcdf file
!
      if (my_pe==0) then 
       print*,' writing time averages to file time_ave.cdf'
       call def_grid_cdf('time_ave.cdf')
       iret=nf_open('time_ave.cdf',NF_WRITE,ncid)
       iret=nf_set_fill(ncid, NF_NOFILL, iret)
       call ncredf(ncid, iret)
       iret=nf_inq_dimid(ncid,'xt',xtdim)
       iret=nf_inq_dimid(ncid,'xu',xudim)
       iret=nf_inq_dimid(ncid,'yt',ytdim)
       iret=nf_inq_dimid(ncid,'yu',yudim)
       iret=nf_inq_dimid(ncid,'zt',ztdim)
       iret=nf_inq_dimid(ncid,'zu',zudim)
       iret=nf_inq_dimid(ncid,'Time',tdim)

       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'um', NCFLOAT,4,dims,iret)
       name='Mean zonal velocity'; unit = 'm/s'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vm', NCFLOAT,4,dims,iret)
       name='Mean meridional velocity'; unit = 'm/s'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,zudim, tdim/)
       varid = ncvdef (ncid,'wm', NCFLOAT,4,dims,iret)
       name='Mean vertical velocity'; unit = 'm/s'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'bm', NCFLOAT,4,dims,iret)
       name='Mean buoyancy'; unit = 'm/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'pm', NCFLOAT,4,dims,iret)
       name='Mean pressure'; unit = 'm^2/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'uu', NCFLOAT,4,dims,iret)
       name='Zonal velocity variance'; unit = 'm^2/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vv', NCFLOAT,4,dims,iret)
       name='Meridional velocity variance'; unit = 'm^2/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'uv', NCFLOAT,4,dims,iret)
       name='Cross velocity variance'; unit = 'm^2/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,zudim, tdim/)
       varid = ncvdef (ncid,'ww', NCFLOAT,4,dims,iret)
       name='Vertical velocity variance'; unit = 'm^2/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'bb', NCFLOAT,4,dims,iret)
       name='Buoyancy variance'; unit = 'm^2/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'b3', NCFLOAT,4,dims,iret)
       name='3. moment buoyancy'; unit = 'm^3/s^6'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'pp', NCFLOAT,4,dims,iret)
       name='Pressure variance'; unit = 'm^4/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

 if (diag_eddy_fluxes) then
       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'ub', NCFLOAT,4,dims,iret)
       name='Zonal eddy buoyancy flux'; unit = 'm^2/s^3'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vb', NCFLOAT,4,dims,iret)
       name='Meridional eddy buoyancy flux'; unit = 'm^2/s^3'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,zudim, tdim/)
       varid = ncvdef (ncid,'wb', NCFLOAT,4,dims,iret)
       name='Vertical eddy buoyancy flux'; unit = 'm^2/s^3'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'ubb', NCFLOAT,4,dims,iret)
       name='Zonal buoyancy variance flux'; unit = 'm^3/s^5'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vbb', NCFLOAT,4,dims,iret)
       name='Meridional buoyancy variance flux'; unit = 'm^3/s^5'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,zudim, tdim/)
       varid = ncvdef (ncid,'wbb', NCFLOAT,4,dims,iret)
       name='Vertical buoyancy variance flux'; unit = 'm^3/s^5'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'ub3', NCFLOAT,4,dims,iret)
       name='Zonal buoyancy 3.order flux'; unit = 'm^4/s^7'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vb3', NCFLOAT,4,dims,iret)
       name='Meridional buoyancy 3. order flux'; unit = 'm^4/s^7'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,zudim, tdim/)
       varid = ncvdef (ncid,'wb3', NCFLOAT,4,dims,iret)
       name='Vertical buoyancy 3. order flux'; unit = 'm^4/s^7'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

 endif

 if (diag_eddy_flux_tendency) then
       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'ugeo1', NCFLOAT,4,dims,iret)
       name='Geostr. flux tendency 1'; unit = 'm^2/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'ugeo2', NCFLOAT,4,dims,iret)
       name='Geostr. flux tendency 2'; unit = 'm^2/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vgeo1', NCFLOAT,4,dims,iret)
       name='Geostr. flux tendency 1'; unit = 'm^2/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vgeo2', NCFLOAT,4,dims,iret)
       name='Geostr. flux tendency 2'; unit = 'm^2/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'uwN', NCFLOAT,4,dims,iret)
       name='Vertical flux tendency'; unit = 'm^2/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vwN', NCFLOAT,4,dims,iret)
       name='Vertical flux tendency'; unit = 'm^2/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)
 endif

 if  (remove_back_stratification) then
       varid = ncvdef (ncid,'b_r', NCFLOAT,1,ztdim,iret)
       name = 'Reference buoyancy profile'; unit = 'm/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)
 endif
       do n=1,nr_tr
        dims = (/xtdim, ytdim,ztdim, tdim/)
        write(name, '("trm",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name = 'Mean tracer'; unit = ' '
        call dvcdf(ncid,varid,name,64,unit,32,spval)

        write(name, '("trvar",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name = 'Tracer variance'; unit = ' '
        call dvcdf(ncid,varid,name,64,unit,32,spval)

        write(name, '("tr3",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name = '3. moment tracer'; unit = ' '
        call dvcdf(ncid,varid,name,64,unit,32,spval)
 if (diag_eddy_fluxes) then
        dims = (/xudim, ytdim,ztdim, tdim/)
        write(name, '("utr",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name='Zonal eddy tracer flux'; unit = 'm/s '
        call dvcdf(ncid,varid,name,64,unit,32,spval)

        dims = (/xtdim, yudim,ztdim, tdim/)
        write(name, '("vtr",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name='Meridional eddy tracer flux'; unit = 'm/s '
        call dvcdf(ncid,varid,name,64,unit,32,spval)

        dims = (/xtdim, ytdim,zudim, tdim/)
        write(name, '("wtr",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name='Vertical eddy tracer flux'; unit = 'm/s '
        call dvcdf(ncid,varid,name,64,unit,32,spval)

        dims = (/xudim, ytdim,ztdim, tdim/)
        write(name, '("utrvar",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name='Zonal tracer variance flux'; unit = ' m/s'
        call dvcdf(ncid,varid,name,64,unit,32,spval)

        dims = (/xtdim, yudim,ztdim, tdim/)
        write(name, '("vtrvar",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name='Meridional tracer variance flux'; unit = 'm/s'
        call dvcdf(ncid,varid,name,64,unit,32,spval)

        dims = (/xtdim, ytdim,zudim, tdim/)
        write(name, '("wtrvar",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name='Vertical tracer variance flux'; unit = 'm/s'
        call dvcdf(ncid,varid,name,64,unit,32,spval)

        dims = (/xudim, ytdim,ztdim, tdim/)
        write(name, '("ut3_",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name='Zonal tracer  3.order flux'; unit = ' m/s'
        call dvcdf(ncid,varid,name,64,unit,32,spval)

        dims = (/xtdim, yudim,ztdim, tdim/)
        write(name, '("vt3_",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name='Meridional tracer 3.order flux'; unit = 'm/s'
        call dvcdf(ncid,varid,name,64,unit,32,spval)

        dims = (/xtdim, ytdim,zudim, tdim/)
        write(name, '("wt3_",i2)') n; call replace_space_zero(name)
        varid = ncvdef (ncid,name, NCFLOAT,4,dims,iret)
        name='Vertical tracer 3.order flux'; unit = 'm/s'
        call dvcdf(ncid,varid,name,64,unit,32,spval)

 endif
       enddo
 if (diag_potential_vorticity) then
       dims = (/xtdim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'qm', NCFLOAT,4,dims,iret)
       name='Ertel PV'; unit = '1/s'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'qq', NCFLOAT,4,dims,iret)
       name='PV variance'; unit = '1/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'q3', NCFLOAT,4,dims,iret)
       name='3. moment pv'; unit = '1/s^3'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'uq', NCFLOAT,4,dims,iret)
       name='Zonal eddy pv flux'; unit = 'm/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vq', NCFLOAT,4,dims,iret)
       name='Meridional eddy pv flux'; unit = 'm/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,zudim, tdim/)
       varid = ncvdef (ncid,'wq', NCFLOAT,4,dims,iret)
       name='Vertical eddy pv flux'; unit = 'm/s^2'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'uqq', NCFLOAT,4,dims,iret)
       name='Zonal pv variance flux'; unit = 'm^2/s^3'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vqq', NCFLOAT,4,dims,iret)
       name='Meridional pv variance flux'; unit = 'm^2/s^3'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,zudim, tdim/)
       varid = ncvdef (ncid,'wqq', NCFLOAT,4,dims,iret)
       name='Vertical pv variance flux'; unit = 'm^2/s^3'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xudim, ytdim,ztdim, tdim/)
       varid = ncvdef (ncid,'uq3', NCFLOAT,4,dims,iret)
       name='Zonal pv 3.order flux'; unit = 'm^3/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, yudim,ztdim, tdim/)
       varid = ncvdef (ncid,'vq3', NCFLOAT,4,dims,iret)
       name='Meridional pv 3. order flux'; unit = 'm^3/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)

       dims = (/xtdim, ytdim,zudim, tdim/)
       varid = ncvdef (ncid,'wq3', NCFLOAT,4,dims,iret)
       name='Vertical pv 3. order flux'; unit = 'm^3/s^4'
       call dvcdf(ncid,varid,name,64,unit,32,spval)
 endif

       iret= nf_put_att_int(ncid,nf_global,'nitts',nf_int,1,nitts)
       call ncendf(ncid, iret)
 if  (remove_back_stratification) then
       iret=nf_inq_varid(ncid,'b_r',varid)
       iret= nf_put_vara_double(ncid,varid,1,nz,b_r)
 endif
       call ncclos (ncid, iret)
      endif ! my_pe ==0
!-----------------------------------------------------------------------
!      write to netcdf file
!-----------------------------------------------------------------------
      do npe=0,n_pes
       if (my_pe==npe) then

         iret=nf_open('time_ave.cdf',NF_WRITE,ncid)
         iret=nf_set_fill(ncid, NF_NOFILL, iret)

         iret=nf_inq_dimid(ncid,'Time',tdim)
         iret=nf_inq_dimlen(ncid,tdim,ilen)
         iret=nf_inq_varid(ncid,'Time',timeid)
         if (my_pe==0) then
          ilen=ilen+1
          time = itt*dt !current_time-initial_time
          fxa = time/86400. ! time%days + time%seconds/86400.
          iret= nf_put_vara_double(ncid,timeid,ilen,1,fxa)
         endif

         Corner = (/1,js_pe,1,ilen/); 
         edges  = (/nx ,je_pe-js_pe+1,nz,1/)
         iret=nf_inq_varid(ncid,'um',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,    umean(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'vm',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,    vmean(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'wm',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,    wmean(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'bm',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,     bmean(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'pm',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,       pmean(:,js_pe:je_pe,:))

         iret=nf_inq_varid(ncid,'uu',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   uvar(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'vv',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,    vvar(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'uv',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   uv(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'ww',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,     wvar(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'bb',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,      bvar(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'b3',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,      b3(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'pp',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,      pvar(:,js_pe:je_pe,:))

 if (diag_eddy_fluxes) then
         iret=nf_inq_varid(ncid,'ub',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,     ub(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'vb',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,     vb(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'wb',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,       wb(:,js_pe:je_pe,:,1))

         iret=nf_inq_varid(ncid,'ubb',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,    ubb(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'vbb',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,     vbb(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'wbb',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   wbb(:,js_pe:je_pe,:,1))

         iret=nf_inq_varid(ncid,'ub3',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,     ub3(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'vb3',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   vb3(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'wb3',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,     wb3(:,js_pe:je_pe,:,1))
 endif

 if (diag_eddy_flux_tendency) then
         iret=nf_inq_varid(ncid,'ugeo1',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,     ugeo1(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'ugeo2',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,    ugeo2(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'vgeo1',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   vgeo1(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'vgeo2',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,    vgeo2(:,js_pe:je_pe,:))

         iret=nf_inq_varid(ncid,'uwN',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,  uwN(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'vwN',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   vwN(:,js_pe:je_pe,:,1))
 endif
        do n=1,nr_tr
         write(name, '("trm",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,    tmean(:,js_pe:je_pe,:,n))

         write(name, '("trvar",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,   tvar(:,js_pe:je_pe,:,n))

         write(name, '("tr3",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,    t3(:,js_pe:je_pe,:,n))
 if (diag_eddy_fluxes) then
          write(name, '("utr",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,  ut(:,js_pe:je_pe,:,1,n))
          write(name, '("vtr",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,  vt(:,js_pe:je_pe,:,1,n))
          write(name, '("wtr",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,   wt(:,js_pe:je_pe,:,1,n))

          write(name, '("utrvar",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,  utt(:,js_pe:je_pe,:,1,n))
          write(name, '("vtrvar",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,  vtt(:,js_pe:je_pe,:,1,n))
          write(name, '("wtrvar",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,    wtt(:,js_pe:je_pe,:,1,n))

          write(name, '("ut3_",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,  ut3(:,js_pe:je_pe,:,1,n))
          write(name, '("vt3_",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,   vt3(:,js_pe:je_pe,:,1,n))
          write(name, '("wt3_",i2)') n; call replace_space_zero(name)
          iret=nf_inq_varid(ncid,name,varid)
          iret= nf_put_vara_double(ncid,varid,corner,edges,     wt3(:,js_pe:je_pe,:,1,n))

 endif
         enddo
 if (diag_potential_vorticity) then
         iret=nf_inq_varid(ncid,'qm',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,  qmean(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'qq',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   qvar(:,js_pe:je_pe,:))
         iret=nf_inq_varid(ncid,'q3',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,    q3(:,js_pe:je_pe,:))

         iret=nf_inq_varid(ncid,'uq',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,  uq(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'vq',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,  vq(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'wq',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   wq(:,js_pe:je_pe,:,1))

         iret=nf_inq_varid(ncid,'uqq',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,  uqq(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'vqq',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,  vqq(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'wqq',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   wqq(:,js_pe:je_pe,:,1))

         iret=nf_inq_varid(ncid,'uq3',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   uq3(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'vq3',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,   vq3(:,js_pe:je_pe,:,1))
         iret=nf_inq_varid(ncid,'wq3',varid)
         iret= nf_put_vara_double(ncid,varid,corner,edges,    wq3(:,js_pe:je_pe,:,1))
 endif
         call ncclos (ncid, iret)
       endif
       call fortran_barrier
      enddo
 end subroutine time_averages_write

