#include "options.inc"

c=======================================================================
c      solve three dimensional Possion equation
c           A * dpsi = forc,  where A = nabla^2  
c      with Neumann boundary conditions
c      used for non-hydrostatic pressure
c=======================================================================

      module congrad3D_module
      implicit none
      private
      public congrad3D
      contains

      subroutine congrad3D(cf, forc,
     &                  max_iterations, iterations, epsilon)
      use cpflame_module
      implicit none
c=======================================================================
c      solve:  A * dpsi = forc
c=======================================================================
      integer :: max_iterations,iterations,i,j,k,n
      real :: epsilon,estimated_error
      real :: forc(imt,jmt,km)
      real :: res(imt,jmt,km),Z(imt,jmt,km),Zres(imt,jmt,km)
      real :: s(imt,jmt,km),As(imt,jmt,km)
      real :: cf(imt,jmt,km,-1:1,-1:1,-1:1)
      real :: zresmax,betakm1,betak,betak_min,betaquot,s_dot_As,smax
      real :: alpha,step,step1,convergence_rate
      logical :: diverged,converged

      Z=0.;Zres=0.;s=0.;As=0.
c-----------------------------------------------------------------------
c     impose boundary conditions on guess
c     dpsi(0) = guess
c-----------------------------------------------------------------------
      call setcyclic3D(psi)
c-----------------------------------------------------------------------
c     make approximate inverse operator Z (always even symmetry)
c-----------------------------------------------------------------------
      call make_inv (cf, Z )
      call setcyclic3D(Z)
c-----------------------------------------------------------------------
c     res(0)  = forc - A * dpsi(0)
c-----------------------------------------------------------------------
      call op9_vec(cf, psi, res)
      do k=1,km
       do j=js_pe,je_pe
        do i=1,imt
         res(i,j,k) = forc(i,j,k) - res(i,j,k)
        enddo
       enddo
      enddo
      call setcyclic3D(res)
c-----------------------------------------------------------------------
c     Zres(k-1) = Z * res(k-1)
c     see if guess is a solution, bail out to avoid division by zero
c-----------------------------------------------------------------------
      n = 0
      diverged=.false.
      call inv_op(Z, res, Zres)
      call setcyclic3D(Zres)
      Zresmax = absmax(Zres)
c       Assume convergence rate of 0.99 to extrapolate error
      if (100.0 * Zresmax .lt. epsilon) then
	  estimated_error = 100.0 * Zresmax 
	  goto 101
      endif
c-----------------------------------------------------------------------
c     beta(0) = 1
c     s(0)    = zerovector()
c-----------------------------------------------------------------------
      betakm1 = 1.0
      s=0.
c-----------------------------------------------------------------------
c     begin iteration loop
c-----------------------------------------------------------------------
      do n = 1,max_iterations
c-----------------------------------------------------------------------
c       Zres(k-1) = Z * res(k-1)
c-----------------------------------------------------------------------
        call inv_op(Z, res, Zres)
        call setcyclic3D(Zres)
c-----------------------------------------------------------------------
c       beta(k)   = res(k-1) * Zres(k-1)
c-----------------------------------------------------------------------
        betak = dot2(Zres, res)
        if (n .eq. 1) then
          betak_min = abs(betak)
	elseif (n .gt. 2) then
          betak_min = min(betak_min, abs(betak))
          if (abs(betak) .gt. 100.0*betak_min) then
           print*,'PE ',my_pe,' : ',
     &      'WARNING: 3D conj. gradient terminated because correction'
     &,     '         steps are diverging. Probable cause...roundoff'
            diverged=.true.
            goto 101
	  endif
	endif
c-----------------------------------------------------------------------
c       s(k)      = Zres(k-1) + (beta(k)/beta(k-1)) * s(k-1)
c-----------------------------------------------------------------------
        betaquot = betak/betakm1
        do k=1,km
         do j=js_pe,je_pe
          do i=1,imt
           s(i,j,k) = Zres(i,j,k) + betaquot * s(i,j,k)
          enddo
         enddo
        enddo
        call setcyclic3D(s)
        call border_exchg3D(s,1)

c-----------------------------------------------------------------------
c       As(k)     = A * s(k)
c-----------------------------------------------------------------------
        call op9_vec(cf, s, As)
        call setcyclic3D(As)
c-----------------------------------------------------------------------
c       If s=0 then the division for alpha(k) gives a float exception.
c       Assume convergence rate of 0.99 to extrapolate error.
c       Also assume alpha(k) ~ 1.
c-----------------------------------------------------------------------
        s_dot_As = dot2(s, As)
        if (abs(s_dot_As) .lt. abs(betak)*1.e-10) then
          smax = absmax(s)
	  estimated_error = 100.0 * smax 
	  goto 101
	endif
c-----------------------------------------------------------------------
c       alpha(k)  = beta(k) / (s(k) * As(k))
c-----------------------------------------------------------------------
        alpha = betak / s_dot_As
c-----------------------------------------------------------------------
c       update values:
c       dpsi(k)   = dpsi(k-1) + alpha(k) * s(k)
c       res(k)    = res(k-1) - alpha(k) * As(k)
c-----------------------------------------------------------------------
        do k=1,km
         do j=js_pe,je_pe
          do i=1,imt
           psi(i,j,k)  = psi(i,j,k) + alpha * s(i,j,k)
           res(i,j,k)  = res(i,j,k) - alpha * As(i,j,k)
          enddo
         enddo
        enddo
        smax = absmax(s)
c-----------------------------------------------------------------------
c       test for convergence
c       if (estimated_error) < epsilon) exit
c-----------------------------------------------------------------------
        step = abs(alpha) * smax
        if (n .eq. 1) then
          step1 = step
          estimated_error = step
          if (step .lt. epsilon) goto 101
        else if (step .lt. epsilon) then
          convergence_rate = exp(log(step/step1)/(n-1))
	  estimated_error = step*convergence_rate/(1.0-convergence_rate)
          if (estimated_error .lt. epsilon) goto 101
        end if
        betakm1 = betak
c        print*,'n=',n,' error=',step,' eps=',epsilon
      end do
c-----------------------------------------------------------------------
c     end of iteration loop
c-----------------------------------------------------------------------
  101 continue
c       print*,' estimated error ',estimated_error
      if ((n .gt. max_iterations).or.(diverged)) then
          print*,'PE ',my_pe,' : 3D-Poisson solver is not converged '
        converged = .false.
      else
        converged = .true.
      end if
      iterations = n
c      if (.not.converged) call halt_stop('  ')
      end subroutine congrad3D




      subroutine op9_vec(cf, p1, res)
      use cpflame_module
      implicit none
c-----------------------------------------------------------------------
c                       res = A * dpsi
c-----------------------------------------------------------------------
      real :: cf(imt,jmt,km,-1:1,-1:1,-1:1) 
      real :: p1(imt,jmt,km), res(imt,jmt,km)
      integer :: i,j,k,ii,jj,kk,js,je
      js =max(2,js_pe);   je  = min(je_pe,jmt-1)
      res=0.
      do kk=-1,1
       do jj=-1,1
        do ii=-1,1
      do k=2,km-1
       do j=js,je
        do i=2,imt-1
          res(i,j,k) = res(i,j,k) + 
     &         cf(i,j,k,ii,jj,kk)*p1(i+ii,j+jj,k+kk) 
        end do
       end do
      end do
        end do
       end do
      end do
      
      end subroutine op9_vec



      subroutine inv_op(Z, res, Zres)
      use cpflame_module
      implicit none
c-----------------------------------------------------------------------
c     apply and approximate inverse Z or the operator A
c-----------------------------------------------------------------------
      real ::  Z(imt,jmt,km),res(imt,jmt,km),Zres(imt,jmt,km)
      integer :: i,j,k
      do k=1,km
       do j=js_pe,je_pe
        do i=1,imt
          Zres(i,j,k) = Z(i,j,k) * res(i,j,k)
        end do
       end do
      end do
      end subroutine inv_op


      subroutine make_inv (cf, Z)
      use cpflame_module
      implicit none
c-----------------------------------------------------------------------
c     construct an approximate inverse Z to A
c-----------------------------------------------------------------------
      real :: cf(imt,jmt,km,-1:1,-1:1,-1:1) 
      real ::  Z(imt,jmt,km)
      integer :: i,j,k
      do k=1,km
       do j=js_pe,je_pe
        do i=1,imt
          if (cf(i,j,k,0,0,0)/=.0) then
            Z(i,j,k) = 1./cf(i,j,k,0,0,0)
          else
            Z(i,j,k)=0.
          endif
        end do
       end do
      end do
      end subroutine make_inv


      real function absmax(p1)
      use cpflame_module
      implicit none
      real :: p1(imt,jmt,km),s
      integer :: i,j,k
      s=0
      do k=2,km-1
       do j=js_pe,je_pe
        do i=2,imt-1
         s = max( abs(p1(i,j,k)*maskT(i,j,k)), s )
        enddo
       enddo
      enddo
      call global_max(s)
      absmax=s
      end function absmax


      real  function dot2(p1,p2)
      use cpflame_module
      implicit none
      real :: p1(imt,jmt,km),p2(imt,jmt,km),s
      integer :: i,j,k
      s=0
      do k=1,km
       do j=js_pe,je_pe
        do i=1,imt
         s = s+p1(i,j,k)*p2(i,j,k)*maskt(i,j,k)
        enddo
       enddo
      enddo
      call global_sum(s)
      dot2=s
      end function dot2

      end module congrad3D_module









