

       module time_manager_module
c
c-------------------------------------------------------------------------
c      use this module as a time manager of a complete integration.
c      time derivatives could be written like:
c      h(::,taup1) = h(::,taum1) + c2dt* forcing(tau,taum1)
c
c      a typical main loop of a model could look like:
c
c       call init_time_manager(dt_in,runlen,snap_int,restart_stamp )
c 10    continue
c 20     call time_step_begin()
c        ... stuff to integrate the model ...
c        if (snapshot_time_step) diagnose the model
c        call time_step_end()
c        if (eulerback2) goto 20    ! do the second part of the euler backw. 
c       if (.not. eorun ) goto 10
c
c                      C.Eden Feb. 2002 (mailto ceden@phys.ocean.dal.ca)
c-------------------------------------------------------------------------
c
       use time_type_module
       implicit none

       integer             :: mixing_freq = 17     ! freq. in time steps of 
                                                   ! mixing time steps
       character (len=80)  :: mixing_type = 'euler_backward' 
                     ! type of mixing time step
                     ! 'euler_backward' and 'euler_foreward' are implemented
                     ! note that euler_backward is more dissipative and
                     ! damps also spatial scales ...
       type( time_type )          :: initial_time    ! the initial time, for which
                                                     ! the model was started
       type( time_type )          :: current_time    ! the current model time
       type( time_type )          :: restart_time    ! the time of the last restart
       type( time_type )          :: stop_time       ! the time of the end of
                                                     ! the present integration
       type( time_type )          :: snapshot_time   ! the time for the next
                                                     ! snapshot to write
       type( time_type )          :: snapshot_inctime! the interval between 
                                                     ! snapshots
       integer          :: initial_year  = 1900      ! initial date
       integer          :: initial_month = 01
       integer          :: initial_day   = 01
       integer          :: initial_hour  = 00
       integer          :: initial_min   = 00
       integer          :: initial_sec   = 00
       real  (kind=8)   :: dt,c2dt           ! time step in secs and 2*dt
                                             ! or 1*dt for an euler step
       integer          :: itt = 1           ! number of time steps
                                             ! in present integration
       integer          :: taum1=0,tau=1,taup1=2 ! pointer to data
                                                 ! which depends on time
           ! tau points to the present time step, taum1, taup1 to the
           ! previous and next time step. For an euler time step, taum1
           ! equals tau.
       integer,private  :: otaum1=0, intermediate ! pointer for eulerbackward
       character        :: current_stamp*32       ! current time stamp 
       logical          :: eulerback1= .false.    ! is this the first part of
                                                  ! an euler backward step?
       logical          :: eulerback2= .false.    ! is this the second part of
                                                  ! an euler backward step?
       logical          :: eulerfore = .false.    ! is this an euler foreward step?
       logical          :: end_of_run           = .false. ! is the end of run reached
       logical          :: last_time_step       = .false. ! is this the last time step
       logical          :: snapshot_time_step   = .false. ! write a snapshots?
       logical          :: end_of_year          = .false. ! end of year
       logical          :: end_of_season        = .false.
       logical          :: end_of_month         = .false.
       logical          :: end_of_day           = .false.
       logical          :: time_manager_verbose = .true.  ! be verbose
       logical          :: tmngr_initialized    = .false.

       contains

       subroutine init_time_manager(dt_in,runlen_in,snap_int_in,
     &                              restart_stamp_in,itt_in)
c
c-------------------------------------------------------------------------
c      initialze the whole time manager
c-------------------------------------------------------------------------
c 
       real(kind=8), intent(in) :: dt_in    ! time step in seconds
       real(kind=8), intent(in) :: runlen_in   ! length of run in days
       real(kind=8), intent(in) :: snap_int_in ! snapshot interval in days
       character, intent(in) ::  restart_stamp_in*32
                                    ! year,months, etc of restart
       integer, intent(in) :: itt_in! counter of time steps
                                    ! of the restart
       type( time_type ):: time  
       integer :: n

       if (.not. tmngr_initialized ) then
         tmngr_initialized = .true.
       endif

!      set the time step counter
       itt=itt_in

!      set the time steps
       dt=dt_in
       c2dt=2.*dt
       if (int(dt)*1. /= dt) then
         print*,' ERROR: use of timesteps with fractional seconds'
         print*,' in init_time_manager'
         stop
       endif

!      default calendar type to be used
       call set_calendar_type(no_leap)

!      set initial time
       initial_time = set_date(initial_year, initial_month, initial_day,
     &                         initial_hour, initial_min, initial_sec)

!      set restart time
       restart_time = get_stamp(restart_stamp_in)

!      update current time
       current_time = restart_time
       call set_stamp(current_stamp, current_time )

!      set end of integration
       call set_time ( time,
     &        nint( (runlen_in-int(runlen_in))*60*60*24 ), 
     &        int(runlen_in))
       stop_time = current_time + time 

!      set next snapshot time
       snapshot_time = initial_time

       call set_time(snapshot_inctime ,
     &        nint( (snap_int_in-int(snap_int_in))*60*60*24 ), 
     &        int(snap_int_in))

       if (snap_int_in>0.) then
 20     snapshot_time = snapshot_time + snapshot_inctime
        if (snapshot_time < current_time ) goto 20
       else
        snapshot_time=current_time
       endif

       if ( time_manager_verbose ) then
        print*,' '
        print*,' setting up the model integration for'
        print*,' ',runlen_in,' days, snapshot interval is ',
     &          snap_int_in,' days'
        print*,' time step will be ',dt,' seconds'
        print*,' '
       endif

       end subroutine init_time_manager


       subroutine time_step_begin()
c
c-------------------------------------------------------------------------
c      set up some things for each time step
c      use this before doing stuff in the present time step
c-------------------------------------------------------------------------
c
       type( time_type ):: time  

       if (mod(itt,mixing_freq) == 0) then ! start a mixing time step

         c2dt  = 1.0*dt

         if     (mixing_type == 'euler_backward') then

          if (eulerback2) then

           eulerback1 = .false. ! second part of euler backward step
           eulerback2 = .true.  ! second part of euler backward step
           eulerfore  = .false.
           c2dt  = 1.0*dt

          else

           eulerback1=.true. ! first part of euler backward step
           eulerback2=.false.
           eulerfore =.false.
           otaum1=taum1           ! we do not need this pointer, but remember it
           taum1=tau              ! set tau and taum1 to the same arrays
                                  ! and compute intermediate step in taup1
                                  ! see in time_step_end what happens next
          endif
         elseif (mixing_type == 'euler_foreward') then

          otaum1=taum1      ! remember that pointer which is not needed here
          taum1=tau         ! simple euler foreward step
          eulerback1=.false.
          eulerback2=.false.
          eulerfore =.true.

         else
          print*,        ' other mixing time steps than'//
     &                   ' euler backward and euler foreward'//
     &                   ' are not implemented so for '
          stop
         endif

       else

         eulerfore =.false.
         eulerback1=.false.
         eulerback2=.false.
         c2dt  = 2.0*dt

       endif

c      is this the last time step ? 

       call set_time(time,nint(dt),0)
       if ( current_time + time >= stop_time) last_time_step = .true.
c      do nothing if this is the first part of an eulerbackward step
       if (eulerback1) last_time_step = .false.

c      is this a snapshot time step ?

       if ( current_time >= snapshot_time .and. .not. eulerback1  ) then 
        snapshot_time_step = .true.
        snapshot_time = snapshot_time + snapshot_inctime
        if (snapshot_time <= current_time) then
         snapshot_time = current_time
        endif
       else
        snapshot_time_step = .false.
       endif

       call set_time(time,nint(dt),0)
       time=current_time+time

c      is this the end of the year ?
       end_of_year= .false.
       if (get_year(time) /= get_year( current_time) )
     &     end_of_year = .true.
       if (eulerback1) end_of_year=.false.

c      is this the end of the season ?
       end_of_season= .false.
       if (get_season_of_year(time) /= get_season_of_year(current_time))
     &     end_of_season = .true.
       if (eulerback1) end_of_season=.false.

c      is this the end of the month ?
       end_of_month= .false.
       if (get_month_of_year(time) /= get_month_of_year(current_time))
     &     end_of_month = .true.
       if (eulerback1) end_of_month=.false.

c      is this the end of the day ?
       end_of_day= .false.
       if (get_day_of_month(time) /= get_day_of_month( current_time))
     &     end_of_day = .true.
       if (eulerback1) end_of_day=.false.

c      implement other switches here

       end subroutine time_step_begin



       subroutine time_step_end()
c
c-------------------------------------------------------------------------
c      increments all pointers and the time
c      use this after done all stuff in the present time step
c-------------------------------------------------------------------------
c

       if (eulerfore) then
c
c       end of an euler foreward step
c
        eulerback1=.false.
        eulerback2=.false.
        eulerfore =.false.
c        taum1=tau   ! was already set to this value
        tau=taup1
        taup1=otaum1
        call time_goes_by()

       elseif (eulerback1) then
c
c       end of the first part of an euler backward step
c
        eulerback1=.false.        ! start now the second part of the backward step
        eulerback2=.true.         ! start now the second part of the backward step
        eulerfore =.false.
        tau=taup1                 ! in taup1 the intermediate step was computed
                                  ! now compute the end result: taum1 
                                  ! stays the same and forcing at tau becomes now
                                  ! the intermediate result, forcing at taum1
                                  ! is still the tau from previous leapfrog step
        intermediate = taup1      ! we have to remember that pointer
        taup1=otaum1              ! store the end result of this step
                                  ! in the so far unused array

       elseif (eulerback2) then
c
c       end of the second part of an euler backward step
c
        eulerback1=.false.
        eulerback2=.false.
        eulerfore =.false.
        tau  =taup1               ! taum1 was unchanged during the backward step
                                  ! and still points to tau of the previous
                                  ! leapfrog step
        taup1=intermediate        ! store future results in an array which is not
                                  ! needed anymore.
        call time_goes_by()

       else
c
c       end of a leapfrog time step
c
        eulerback1=.false.
        eulerback2=.false.
        eulerfore =.false.

        otaum1=taum1
        taum1= tau
        tau  = taup1
        taup1= otaum1
        call time_goes_by()
       endif

       end subroutine time_step_end


       subroutine time_goes_by
c
c-------------------------------------------------------------------------
c      as times goes by
c-------------------------------------------------------------------------
c
       type( time_type ):: time  
       itt=itt+1
       call set_time(time,nint(dt),0)
       current_time = current_time + time 
       call set_stamp(current_stamp, current_time )
       if ( current_time >= stop_time) end_of_run = .true.
       end subroutine time_goes_by

       subroutine set_stamp (stamp, time)
c
c-------------------------------------------------------------------------
c      converts time to character string
c-------------------------------------------------------------------------
c
       implicit none
       type( time_type ), intent(in) :: time
       character, intent(out) :: stamp*32
c     author:       R.C. Pacanowski     e-mail=>  rcp@gfdl.gov
       integer year, month, day, hour, min, sec
       call get_date(time,year,month,day,hour,min,sec)
       if (year .le. 9999) then
         write (stamp,'(a6,i2,a1,i2,a1,i4,a8,i2,a1,i2,a1,i2)') 
     &                'm/d/y=',month,'/',day,'/'
     &,                year,', h:m:s=', hour,':', min,':', sec
       elseif (year .ge. 10000 .and. year .le. 99999) then
         write (stamp,'(a6,i2,a1,i2,a1,i5,a7,i2,a1,i2,a1,i2)')
     &            'm/d/y=',month,'/',day,'/'
     &,           year,',h:m:s=', hour,':', min,':', sec
       elseif (year .ge. 100000 .and. year .le. 999999) then
         write (stamp,'(a6,i2,a1,i2,a1,i6,a6,i2,a1,i2,a1,i2)')
     &            'm/d/y=',month,'/',day,'/'
     &,           year,'h:m:s=', hour,':',  min,':', sec
       else
         write (*,*) '=>Error: year=',year,',is too large in set_stamp.'
         print*,' in set_stamp'
         stop
       endif
       end subroutine set_stamp


       function get_stamp (stamp)
c
c-------------------------------------------------------------------------
c       converts character string to time type
c-------------------------------------------------------------------------
c
       implicit none
       type( time_type ) :: get_stamp
       character, intent(in) ::  stamp*32
       integer   year, month, day, hour, min, sec
       character skip6*6, skip7*7, skip8*8, skip1*1
c      author:       R.C. Pacanowski     e-mail=>  rcp@gfdl.gov
c
c      for years: 0..9999
       if (stamp(17:17) .eq. ',') then 
        read  (stamp, '(a6,i2,a1,i2,a1,i4,a8,i2,a1,i2,a1,i2)')
     &      skip6, month, skip1, day, skip1, year, skip8, hour
     &,     skip1, min, skip1, sec
c      for years:  10000..99999
       elseif (stamp(18:18) .eq. ',') then
        read  (stamp, '(a6,i2,a1,i2,a1,i5,a7,i2,a1,i2,a1,i2)')
     &      skip6, month, skip1, day, skip1, year, skip7, hour
     &,     skip1, min, skip1, sec
c      for years:  100000..999999
       elseif (stamp(19:19) .eq. 'h') then
        read  (stamp, '(a6,i2,a1,i2,a1,i6,a6,i2,a1,i2,a1,i2)')
     &      skip6, month, skip1, day, skip1, year, skip6, hour
     &,     skip1, min, skip1, sec
       else
        write (*,*) '=>Error: year is too large in get_stamp.'
        print*,' in get_stamp'
        stop
       endif
       get_stamp = set_date(year,month,day,hour,min,sec)
       end function get_stamp

       function get_current_time()
c-------------------------------------------------------------------------
c      get the current model time
c-------------------------------------------------------------------------
       type (time_type) :: get_current_time
       get_current_time = current_time
       end function get_current_time

       function get_stop_time()
c-------------------------------------------------------------------------
c      get the time of the en of run
c-------------------------------------------------------------------------
       type (time_type) :: get_stop_time
       get_stop_time = stop_time
       end function get_stop_time

       function get_itts_til_eorun()
c-------------------------------------------------------------------------
c      get number of time steps til end of run
c-------------------------------------------------------------------------
       integer :: get_itts_til_eorun
       type (time_type) :: time
       call set_time(time,nint(dt),0)
       get_itts_til_eorun=(stop_time - current_time)/time
       end function get_itts_til_eorun

       function get_initial_time()
c-------------------------------------------------------------------------
c      get the initial model time
c-------------------------------------------------------------------------
       type (time_type) :: get_initial_time
       get_initial_time = initial_time
       end function get_initial_time

       function get_month_of_year(time)
       integer get_month_of_year
       type (time_type) :: time
       integer year, month, day, hour, min, sec
       call get_date(time,year,month,day,hour,min,sec)
       get_month_of_year=month
       end function get_month_of_year

       function get_day_of_month(time)
       integer get_day_of_month
       type (time_type) :: time
       integer year, month, day, hour, min, sec
       call get_date(time,year,month,day,hour,min,sec)
       get_day_of_month = day
       end function get_day_of_month

       function get_season_of_year(time)
       integer get_season_of_year
       type (time_type) :: time
       integer year, month, day, hour, min, sec
       call get_date(time,year,month,day,hour,min,sec)
       if (month >=1 .and. month <= 3 ) get_season_of_year=1 ! J/F/M
       if (month >=4 .and. month <= 6 ) get_season_of_year=2 ! A/M/J
       if (month >=7 .and. month <= 9 ) get_season_of_year=3 ! J/A/S
       if (month >=10.and. month <= 12) get_season_of_year=4 ! O/N/D
       end function get_season_of_year

       function get_year(time)
       integer get_year
       type (time_type) :: time
       integer year, month, day, hour, min, sec
       call get_date(time,year,month,day,hour,min,sec)
       get_year=year
       end function get_year

       end module time_manager_module



