c
c-----------------------------------------------------------------------
c     file contains the initialization part of BARBI
c
c-----------------------------------------------------------------------
c

      subroutine setup
c=======================================================================
c     set up everything for BARBI which must be done only once per run
c
c                      C.Eden Feb. 2002 (mailto ceden@phys.ocean.dal.ca)
c=======================================================================
      use barbi_module
      implicit none
      integer i,j,it,n,jm1,jp1,ip1,itt_in
      real (kind=8) :: dfdy,y,maxlayer,maxdxu,degtm,radian
      real (kind=8) :: area,sum

      print*,' '
      print*,'########################################################'
      print*,' '
      print'("           BARBI version ",f5.3)',version
      print*,' '
      print*,'This code is a numerical implementation of the BARBI'
      print*,'(BARotropic-Baroclinic-Interaction) model.'
      print*,'There is no restriction to usage, feel free to test.'
      print*,'However, this software is provided with absolutely '
      print*,'                 NO WARRANTY.'
      print*,'A description and evaluation of BARBI can be found '
      print*,'in Olbers and Eden, 2002, submitted to JPO.'
      print*,'Documentation and code can be downloaded at'
      print*,'http://www.phys.ocean.dal.ca/~ceden/BARBI.html'
      print*,'Note that this code is mainly based on GFDL MOM 2.1,'
      print*,'from which most of the structure and numerics are taken.'
      print*,'########################################################'
      print*,' '
      print*,' reading namelist and allocating memory'
c
      call init_barbi_module
c
      print*,' using ',(n_order+1)/2,' vertical moments'
      print*,' horizontal viscosity   = ',Am,' m^2/s'
      print*,' horizontal diffusivity = ',Ah,' m^2/s'
      print*,' Stability frequency    = ',N0,' 1/s'
c
c=======================================================================
c     some constants
c=======================================================================
c
      radius   = 6370.0e3
      pi       = 3.14159265358979323846264338327950588
      radian   = 360./(2.*pi)
      omega    = pi/43082.0
c
c=======================================================================
c     other parameter
c=======================================================================
c
      N0sqr    = N0**2
      rho0     = 1035.0
      h_0      = 5500.
c
c=======================================================================
c     set truncation parameter
c=======================================================================
c
      call set_gamma()
c
c=======================================================================
c     set up the grids in x (longitude), y (latitude)
c     corresponding to Arakawa "b" grid system
c=======================================================================
c
      print*,' generating an Arakawa B-grid '
c
c-----------------------------------------------------------------------
c     calculate coordinates for "t" and "u" grid cells.
c-----------------------------------------------------------------------
c
      call simple_grid()
c
c-----------------------------------------------------------------------
c     convert grid resolution to cm
c-----------------------------------------------------------------------
c
      radian   = 360./(2.*pi)
      degtm    = radius/radian
      dyt(:) = dytdeg(:)*degtm
      dyu(:) = dyudeg(:)*degtm
      dxt(:) = dxtdeg(:)*degtm
      dxu(:) = dxudeg(:)*degtm

      if (cyclic) then
       dxt(1)   = dxt(imt-1)
       dxt(imt) = dxt(2)
       dxu(1)   = dxu(imt-1)
       dxu(imt) = dxu(2)
      endif
c
c-----------------------------------------------------------------------
c     compute all quantities derived from the grid spacings
c-----------------------------------------------------------------------
c
      if (enable_beta_plane .or. enable_f_plane) then
       phi(:)   = yu(jlat0)/radian
       phit(:)  = yt(jlat0)/radian
      else
       phi(:)   = yu(:)/radian
       phit(:)  = yt(:)/radian
      endif

      do j=1,jmt
        dytr(j)  = 1.0/dyt(j)
        dyt2r(j) = 0.5/dyt(j)
        dyur(j)  = 1.0/dyu(j)
        dyu2r(j) = 0.5/dyu(j)
        cst(j)   = cos(phit(j))
        csu(j)   = cos(phi (j))
        sine(j)  = sin(phi(j))
        cstr(j)     = 1.0/cst(j)
        csur(j)     = 1.0/csu(j)
        tng(j)      = sine(j)/csu(j)
	cstdytr(j)  = 1.0/(cst(j)*dyt(j))
	cstdyt2r(j) = cstdytr(j)*0.5
        csudyur(j)  = 1.0/(csu(j)*dyu(j))
        csudyu2r(j) = 0.5/(csu(j)*dyu(j))
        cst_dytr(j) = cst(j)/dyt(j)
        csu_dyur(j) = csu(j)/dyu(j)
      enddo
      do i=1,imt
        dxtr(i)  = 1.0/dxt(i)
        dxt2r(i) = 0.5/dxt(i)
        dxur(i)  = 1./dxu(i)
        dxu2r(i) = .5/dxu(i)
      enddo
      do i=2,imt-1
        dxmetr(i) = 1.0/(dxt(i) + dxt(i+1))
      enddo
      do i=1,imt
        duw(i) = (xu(i) - xt(i))*degtm
      enddo
      do i=1,imt-1
	due(i) = (xt(i+1) - xu(i))*degtm
      enddo
      if (cyclic) then
       due(imt) = due(2)
      else
       due(imt) = due(imt-1)
      endif
      do j=1,jmt
        dus(j) = (yu(j) - yt(j))*degtm
      enddo
      do j=1,jmt-1
	dun(j) = (yt(j+1) - yu(j))*degtm
      enddo
      dun(jmt) = dun(jmt-1)

      print*,' grid generation done '
c
c=======================================================================
c     generate a topography
c=======================================================================
c
      call simple_topog()
c
c     close basin
c
      h(:,1)=0.
      h(:,jmt)=0.
      h(1,:)=0.
      h(imt,:)=0.
c
c     at open boundaries open basin
c
      if (enable_obc_north) h(:,jmt)=h(:,jmt-1)
      if (enable_obc_south) h(:,1)  =h(:,2)
      if (enable_obc_west)  h(1,:)  =h(2,:)
      if (enable_obc_east)  h(imt,:)=h(imt-1,:)

      if (cyclic) then
       h(1,:)   = h(imt-1,:)
       h(imt,:) = h(2,:)
      endif

      hu=0.
      do j=1,jmt-1
        do i=1,imt-1
           hu(i,j)=( h(i,j)+h(i+1,j)+h(i,j+1)+h(i+1,j+1) )/4.0
c           hu(i,j)=min(h(i,j),h(i+1,j),h(i,j+1),h(i+1,j+1) )
        enddo
      enddo
      if (cyclic) then
       hu(1,:)   = hu(imt-1,:)
       hu(imt,:) = hu(2,:)
      endif

      tmask=1.
      where( h == 0.) tmask = 0.
      umask=0.
      do j=1,jmt-1
        do i=1,imt-1
         umask(i,j)=tmask(i,j)*tmask(i+1,j)*tmask(i,j+1)*tmask(i+1,j+1)
        enddo
      enddo
      if (cyclic) then
       umask(1,:)   = umask(imt-1,:)
       umask(imt,:) = umask(2,:)
      endif
      hu=hu*umask

      hr=0.; hur=0.
      do j=1,jmt
       do i=1,imt
        if (h(i,j) /=0.) hr(i,j) =1./h(i,j)
        if (hu(i,j)/=0.) hur(i,j)=1./hu(i,j)
       enddo
      enddo
c
c=======================================================================
c     Initialize the poisson solver
c=======================================================================
c
      call init_congrad(imt,jmt,tmask,
     & cyclic, enable_obc_north,enable_obc_south,
     & enable_obc_west,enable_obc_east) 
c
c=======================================================================
c     initial conditions
c=======================================================================
c
      if (read_restart) then
        call read_rest(itt_in)
      else
        call init_cond(itt_in)
      endif
c
c=======================================================================
c     set up the time manager
c=======================================================================
c
      call init_time_manager(time_step,runlen,snapint,
     &          'm/d/y= 1/ 1/1900, h:m:s= 0: 0: 0',itt_in) 
c
c=======================================================================
c     wind forcing
c=======================================================================
c
      call simple_wind
c
c     remove basin mean zonal wind stress
c
c      area=0.; sum=0.
c      do i=1,imt
c       do j=1,jmt
c         sum=sum+wind(i,j,1)*dxu(i)*csu(j)*dyu(j)*umask(i,j)
c         area=area+dxu(i)*csu(j)*dyu(j)*umask(i,j)
c       enddo
c      enddo
c      sum=sum/area
c      wind(:,:,1)=wind(:,:,1)-sum
c
      if (cyclic) then
       wind(1,:,:)   = wind(imt-1,:,:)
       wind(imt,:,:) = wind(2,:,:)
      endif
c
c=======================================================================
c     coriolis factors
c=======================================================================
c
      if (enable_beta_plane .or. enable_f_plane) then
       if (jlat0.gt.jmt) then
        print*,' error: jlat0 greater than jmt '
        stop
       endif
       if (jlat0.lt.1) then
        print*,' error: jlat0 less than 1 '
        stop
       endif
      endif

      if (enable_beta_plane) then
c
c     beta plane with f = f0 + beta*y where f0 is at yu(1)
c     if f_plane then beta = 0
c
       dfdy   = 2.*omega*csu(jlat0)/radius
       do j=1,jmt
        y            = (yu(j)-yu(jlat0))*degtm
        cori(:,j,1) = 2.0*omega*sine(jlat0)  + y*dfdy
        cori(:,j,2) = -(2.0*omega*sine(jlat0) + y*dfdy)
       enddo
       print*,' using a beta plane approximation with f_0=',
     &    2*omega*sine(jlat0),' 1/s '
       print*,' and beta = ',dfdy,' 1/s/m'

      elseif (enable_f_plane) then
       cori(:,:,1) = 2.0*omega*sine(jlat0)  
       cori(:,:,2) = -(2.0*omega*sine(jlat0))
       print*,' using a f-plane approximation with f_0=',
     &    2*omega*sine(jlat0),' 1/s '
      else
       do j=1,jmt
        cori(:,j,1) = 2.0*omega*sine(j)
        cori(:,j,2) = -2.0*omega*sine(j)
       enddo
      endif
c
c=======================================================================
c     metric diffusion factors
c=======================================================================
c
      if (enable_beta_plane .or. enable_f_plane) then
c
c       set metric factors = 0 on beta and f plane
c       (strictly, they should be non zero... but they are small)
c
	am3(:)   = 0.
	am4(:,1) = 0.
	am4(:,2) = 0.
      else
       do j=1,jmt
	am3(j)   = am*(1.0-tng(j)*tng(j))/(radius**2)
	am4(j,1) = -am*2.0*sine(j)/(radius*csu(j)*csu(j))
	am4(j,2) = -am4(j,1)
       enddo
      endif
c
c=======================================================================
c     set horizontal mixing coeffs on north and east face of "t" and
c     "u" cells.
c=======================================================================
c
        do j=1,jmt
	  jm1 = max(1,j-1)
	  jp1 = min(jmt,j+1)
	  amc_north(j) = am*cst(jp1)*dytr(jp1)*csur(j)*dyur(j)
	  amc_south(j) = am*cst(j)*dytr(j)*csur(j)*dyur(j)
        enddo
c
        do j=1,jmt
	  jm1 = max(1,j-1)
	  jp1 = min(jmt,j+1)
          ahc_north(j) = ah*csu(j)*dyur(j)*cstr(j)*dytr(j)
          ahc_south(j) = ah*csu(jm1)*dyur(jm1)*cstr(j)*dytr(j)
        enddo
c
c-----------------------------------------------------------------------
c     build coefficients to minimize advection and diffusion computation 
c-----------------------------------------------------------------------
c
      do j=1,jmt
        do i=1,imt
	  ip1 = min(imt,i+1)
	  csudxur(i,j)     = csur(j)*dxur(i)
	  csudxu2r(i,j)    = csur(j)*dxur(i)*0.5
	  am_csudxtr(i,j)  = am*csur(j)*dxtr(ip1)
	enddo
      enddo

      do j=1,jmt
        do i=1,imt
	  cstdxtr(i,j)    = cstr(j)*dxtr(i)
	  cstdxt2r(i,j)   = cstr(j)*dxtr(i)*0.5
	  ah_cstdxur(i,j) = ah*cstr(j)*dxur(i)
	enddo
      enddo
c
c-----------------------------------------------------------------------
c     Some checks
c-----------------------------------------------------------------------
c
      print*,' Checking diffusive criteria'
      do j=2,jmt-1
       do i=2,imt-1
         if (dt > (dxt(i)*cst(j))**2/(2.*ah) ) then
           print*,' criterium exceeded for T grid at i=',i,' j=',j
           print*,' dt = ',dt
           print*,' dx^2/2/ah=', (dxt(i)*cst(j))**2/(2.*ah) 
         endif
         if (dt > (dxu(i)*csu(j))**2/(2.*am) ) then
           print*,' criterium exceeded for U grid at i=',i,' j=',j
           print*,' dt = ',dt
           print*,' dx^2/2/am=', (dxu(i)*csu(j))**2/(2.*am) 
         endif
       enddo
      enddo
      print*,' Ok,done'

      print*,' Checking Munck criteria'
      print*,' viscosity = ',am
      maxlayer=0.;maxdxu=0.
      do j=2,jmt-1
       do i=2,imt-1
         dfdy   = 2.*omega*csu(jlat0)/radius
         if  ( (Am/dfdy)**(1./3.) <= 0.5*dxu(i)*csu(j) ) then
          print*,' criterium exceeded for dxu at i=',i,' j=',j
          print*,' viscous boundary layer: ',  (Am/dfdy)**(1./3.) 
          print*,' dxu = ', 0.5*dxu(i)*csu(j) 
         endif
         maxlayer=max(maxlayer,(Am/dfdy)**(1./3.) )
         maxdxu = max(maxdxu,dxu(i)*csu(j) )
         if  ( (Am/dfdy)**(1./3.) <= 0.5*dyu(j) ) then
          print*,' criterium exceeded for dyu at i=',i,' j=',j
          print*,' viscous boundary layer: ',  (Am/dfdy)**(1./3.) 
          print*,' dyu = ', 0.5*dyu(j) 
         endif
       enddo
      enddo
      print*,' maximal width of viscous boundary layer: ',
     &    maxlayer/1000.,' km'
      print*,' maximal grid spacing: ', maxdxu/1000.,' km'
      print*,' Ok,done'

      call diagi

      print*,' starting now integration'
      print*,''
      end subroutine setup




      subroutine set_gamma
c=======================================================================
c     Set model parameter gamma for different orders
c
c                      C.Eden Feb. 2002 (mailto ceden@phys.ocean.dal.ca)
c=======================================================================
      use barbi_module
      implicit none
      gamma=0.
      if (n_order == 1 ) then
           gamma(:,:,1) = 1.-6./pi**2    
      elseif (n_order == 3 ) then
c           gamma(:,:,1) =-0.10826408947504 
c           gamma(:,:,3) = 0.80027509322268
           gamma(:,:,1) = 0           
           gamma(:,:,3) = 0.66           
           gscal(1,:)=(/1., 0./)
           gscal(2,:)=(/0., 1./)
      elseif (n_order == 5 ) then
           gamma(:,:,1) =  0.02617392756248
           gamma(:,:,3) = -0.37962102159160
           gamma(:,:,5) = 1.20782633761785
c           gamma(:,:,1) = 0. 
c           gamma(:,:,3) = 0. 
c           gamma(:,:,5) = 0.7
       gscal(1,:)=(/-0.506168,0.805012,-0.309433/)
       gscal(2,:)=(/-0.146802,0.726354,-0.671461/)
       gscal(3,:)=(/0.084242,-0.622619,0.777978/)

      elseif (n_order == 9 ) then

       gamma(:,:, 1)=1.264378050708558909320800012210e-03
       gamma(:,:, 3)=-4.505151671118934331161653972231e-02
       gamma(:,:, 5)=4.100221951697804456671292427927e-01
       gamma(:,:, 7)=-1.416739461175453129726520273834e+00
       gamma(:,:, 9)=2.020904214649291930072649847716e+00


       gscal(1,:)=(/4.767957626313863439548867972917e-01,
     &              -7.842555971946739701294859514746e-01,
     &              3.866324141632892508191332581191e-01,
     &              -8.953433315790049196181143997819e-02,
     &              1.038934597638399050245627819322e-02
     &              /)
       gscal(2,:)=(/5.689575533285886471679404507995e-02,
     &              -3.714216676896155422404888213350e-01,
     &              7.064231900167223798803206591401e-01,
     &              -5.716385677870965409397285839077e-01,
     &              1.816712476444269641628892486551e-01
     &              /)
       gscal(3,:)=(/-1.089179533480494170560870514919e-02,
     &              1.484898937893875670024357305010e-01,
     &              -5.433905864971863008960895058408e-01,
     &              7.480646815362848034425269361236e-01,
     &              -3.506537080668933881177906641824e-01
     &              /)
       gscal(4,:)=(/4.445393116555331966543995747543e-03,
     &              -8.872678556154334206240719140624e-02,
     &              4.432565403964531247638092281704e-01,
     &              -7.775283398476003915433807378577e-01,
     &              4.371282605066360571477446228528e-01
     &              /)
       gscal(5,:)=(/2.837266137131796549936124662850e-03,
     &              -6.790055247448924247155588318492e-02,
     &              3.937578805659827096974368032534e-01,
     &              -7.793290157035472898172656641691e-01,
     &              4.826825889435536476312904596853e-01
     &              /)

      elseif (n_order == 21 ) then

       gamma(:,:, 1)=-1.007366915315582867052057736146e-05
       gamma(:,:, 3)=4.109507171889958954125177115202e-06
       gamma(:,:, 5)=5.372755190933276026044040918350e-04
       gamma(:,:, 7)=-1.044804617777117528021335601807e-02
       gamma(:,:, 9)=1.070889332186197862029075622559e-01
       gamma(:,:,11)=-6.482462570420466363430023193359e-01
       gamma(:,:,13)=2.445219248300418257713317871094e+00
       gamma(:,:,15)=-5.884616828523576259613037109375e+00
       gamma(:,:,17)=9.011271761497482657432556152344e+00
       gamma(:,:,19)=-8.476317893015220761299133300781e+00
       gamma(:,:,21)=4.455330321332439780235290527344e+00

       gscal(1,:)=(/-4.766333054316663764282679949247e-01,
     &              7.840303614990239422510853728454e-01,
     &              -3.869034753219216526964885360940e-01,
     &              9.091867244841354833706503768553e-02,
     &              -1.246293513509425660013008041460e-02,
     &              1.118220357777357717998789610192e-03,
     &              -7.074610260577780004691283988194e-05,
     &              3.324924942377156855731566015288e-06,
     &              -1.206327511523451644609954266571e-07,
     &              3.468879785648049990230378501234e-09,
     &              -7.499794573225539046488450824963e-11
     &              /)
       gscal(2,:)=(/4.920809496632456286890899832542e-02,
     &              -3.237762871012418997018755817408e-01,
     &              6.391087733667596415543243892898e-01,
     &              -6.007381627561867443176879532984e-01,
     &              3.293915042102132373358358563564e-01,
     &              -1.182165510265644065013290742172e-01,
     &              2.991549037816711859028018238860e-02,
     &              -5.621019389710830115036710452614e-03,
     &              8.114508324555442944983618680510e-04,
     &              -8.953875774933065770540274730394e-05,
     &              6.245368871073842433202464446973e-06
     &              /)
       gscal(3,:)=(/-3.537841702753233980027758320830e-03,
     &              5.237564775328900495976469642301e-02,
     &              -2.326170250847167664254300234461e-01,
     &              4.919635283798825531320630943810e-01,
     &              -6.069176314914246361809091467876e-01,
     &              4.899847030449622220515948356478e-01,
     &              -2.785854029948041654662915789231e-01,
     &              1.168453638404886169022134367879e-01,
     &              -3.663956314039835959750490701481e-02,
     &              8.088031528491560276372673854439e-03,
     &              -9.598417175623108000062644329375e-04
     &              /)
       gscal(4,:)=(/-2.264602583601552879700308240629e-04,
     &              5.960208295300866050692167874558e-03,
     &              -4.705818536909680410706968700651e-02,
     &              1.768979907256506556301189903024e-01,
     &              -3.876353660115939292474251942622e-01,
     &              5.543906626655306091322472639149e-01,
     &              -5.533061293401139169034763654054e-01,
     &              3.970929105696733851793567282584e-01,
     &              -2.014788253783400540086745422741e-01,
     &              6.569570809984573434014976101025e-02,
     &              -1.033311848118419890019126228253e-02
     &              /)
       gscal(5,:)=(/-1.709417501734237756138776176407e-05,
     &              7.030296517480324764576282703388e-04,
     &              -8.665529447586937938918794088750e-03,
     &              5.075712097302862574332849021630e-02,
     &              -1.723957479717934115903688052640e-01,
     &              3.773395335489777013826540041919e-01,
     &              -5.615186457055946211980312909873e-01,
     &              5.747882708476390600793592966511e-01,
     &              -3.912925193847562876925394448335e-01,
     &              1.599102714757885956853300513103e-01,
     &              -2.961139646000881045506680777635e-02
     &              /)
       gscal(6,:)=(/2.048730612257196100626155041424e-06,
     &              -1.214581589245340297218700431614e-04,
     &              2.139468431942024954917069834437e-03,
     &              -1.774432065659249657674223499271e-02,
     &              8.392936555197183512433412033715e-02,
     &              -2.490950333428795793189181040361e-01,
     &              4.843578028528425782894828444114e-01,
     &              -6.198095006828926445408001200121e-01,
     &              5.038552606317892035292516084155e-01,
     &              -2.358313656671455205149356970651e-01,
     &              4.832409280025098952116024975112e-02
     &              /)
       gscal(7,:)=(/4.190552744338962840187502289363e-07,
     &              -3.402894164164750265017672647971e-05,
     &              7.907759829351083784299003553997e-04,
     &              -8.480710227525450548013985496709e-03,
     &              5.055383293578482456487321883287e-02,
     &              -1.834371204694484058794756720090e-01,
     &              4.221856159220622739525197175681e-01,
     &              -6.196636086478866101145968059427e-01,
     &              5.617490366754943620364315393090e-01,
     &              -2.863638687784754166543166320480e-01,
     &              6.271089344747664273693743552940e-02
     &              /)
       gscal(8,:)=(/1.358854552160842573633255277177e-07,
     &              -1.466956897167202101888924814510e-05,
     &              4.138808122967485157890632496702e-04,
     &              -5.265567256170855005292796846561e-03,
     &              3.637952928468268787298711686162e-02,
     &              -1.495430981204750875424025480243e-01,
     &              3.817448298641363813032967300387e-01,
     &              -6.099738090136147539155331287475e-01,
     &              5.925189718456234100329993452760e-01,
     &              -3.194526032808618354330576494249e-01,
     &              7.320951708619256870225200373170e-02
     &              /)
       gscal(9,:)=(/-6.008078616588443958376920947181e-08,
     &              8.545770249109828364548341328444e-06,
     &              -2.732709567183357296342449682669e-04,
     &              3.870483783061490299509666002109e-03,
     &              -2.931506535301060117904015100976e-02,
     &              1.302812975032291153976160558159e-01,
     &              -3.551979287182689404289703816175e-01,
     &              5.998346886251199050477111995860e-01,
     &              -6.103636254931751325969457866449e-01,
     &              3.421625948572240560885404647706e-01,
     &              -8.103169861921039052177206940542e-02
     &              /)
       gscal(10,:)=(/-3.306964349194005286830706780155e-08,
     &              6.124220570863320999856725668220e-06,
     &              -2.109658462382854159783418523944e-04,
     &              3.184574328450471159684953548208e-03,
     &              -2.549404011928038671364049605472e-02,
     &              1.188784716668503660397249177549e-01,
     &              -3.378867811908940477216845010844e-01,
     &              5.915415188884204367170127625286e-01,
     &              -6.210131437087564876264877966605e-01,
     &              3.576875613024279587648379674647e-01,
     &              -8.672493985543140715410714847167e-02
     &              /)
       gscal(11,:)=(/2.013768736773791894064614821369e-08,
     &              -4.895900477844647632477085436031e-06,
     &              1.768201889749106755101576426981e-04,
     &              -2.780810253759309451371262156272e-03,
     &              2.309041058221536490968617272301e-02,
     &              -1.112351459424176652879268090146e-01,
     &              3.254692942763585117305069616123e-01,
     &              -5.847174992886159383687072477187e-01,
     &              6.281441107741601026503985849558e-01,
     &              -3.692960849117133403929358337336e-01,
     &              9.119415183460717611030332818700e-02
     &              /)

      else
        print*,' you have to find a proper truncation parameter'
        stop
      endif

      end subroutine set_gamma
